package ru.yandex.partner.core.entity.block.type.siteversionandcontextpage;

import java.util.HashSet;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.block.model.BlockWithSiteVersionAndContextPage;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockDefectIds;
import ru.yandex.partner.core.entity.block.service.validation.defects.BlockSimpleDefectParam;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.partner.core.entity.block.model.prop.BlockWithSiteVersionSiteVersionPropHolder.SITE_VERSION;

@ParametersAreNonnullByDefault
@Component
public class BlockWithSiteVersionAndContextPageValidationProvider {
    private final SiteVersionAvailabilityService siteVersionAvailabilityService;

    @Autowired
    public BlockWithSiteVersionAndContextPageValidationProvider(
            SiteVersionAvailabilityService siteVersionAvailabilityService) {
        this.siteVersionAvailabilityService = siteVersionAvailabilityService;
    }

    public <M extends BlockWithSiteVersionAndContextPage> Validator<M, Defect> validator() {
        return block -> {
            var validationBuilder = ModelItemValidationBuilder.of(block);
            if (Objects.nonNull(block.getMultistate()) && block.getMultistate().test(BlockStateFlag.DELETED)) {
                validationBuilder.item(SITE_VERSION).check(
                        inSet(siteVersionAvailabilityService.getAvailableVersionsLiterals(block.getClass())),
                        new Defect<>(BlockDefectIds.SiteVersionDefectId.INVALID_SITE_VERSION,
                                new BlockSimpleDefectParam().withParam(block.getSiteVersion())));
            } else {

                if (Objects.isNull(block.getPageId())) {
                    throw new IllegalArgumentException("page id is null");
                }
                var page = block.getCampaign();

                if (Objects.isNull(page)) {
                    throw new IllegalArgumentException(String.format("Page %s not found", block.getPageId()));
                }
                var featureAndOptions = new HashSet<>(page.getOwner().getFeatures());
                var dependedVersions = siteVersionAvailabilityService
                        .getSiteVersionsThatDependOnFeature(block.getClass(), featureAndOptions);
                var availableBlockVersions = siteVersionAvailabilityService.getBlockSiteVersions(
                        page,
                        dependedVersions,
                        dependedVersions.contains(SiteVersionType.TURBO_DESKTOP.getLiteral()),
                        block.getClass()
                );
                validationBuilder.item(SITE_VERSION).check(inSet(availableBlockVersions),
                        new Defect<>(BlockDefectIds.SiteVersionDefectId.UNAVAILABLE_SITE_VERSION,
                                new BlockSimpleDefectParam().withParam(block.getSiteVersion())));
            }

            return validationBuilder.getResult();
        };
    }

}
