package ru.yandex.partner.core.entity.block.type.siteversionandcontextpage;

import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.page.model.PageWithAllowedAmp;
import ru.yandex.partner.core.entity.page.model.PageWithAllowedTurbo;

import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.AMP;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.DESKTOP;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FLOORAD;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_FULLSCREEN;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.MOBILE_REWARDED;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.TURBO;
import static ru.yandex.partner.core.entity.block.type.commonshowvideoandstrategy.SiteVersionType.TURBO_DESKTOP;

@ParametersAreNonnullByDefault
@Component
public class SiteVersionAvailabilityService {
    private final Map<Class<? extends BaseBlock>, Set<String>> siteVersionsThatDependOnFeature = Map.of(
            RtbBlock.class, SiteVersionType.literals(TURBO_DESKTOP, MOBILE_FULLSCREEN, MOBILE_REWARDED, MOBILE_FLOORAD),
            InternalRtbBlock.class, SiteVersionType.literals(MOBILE_FULLSCREEN, MOBILE_REWARDED, MOBILE_FLOORAD)
    );

    private final SiteVersionAvailabilityFacade siteVersionAvailabilityFacade;

    public SiteVersionAvailabilityService(SiteVersionAvailabilityFacade siteVersionAvailabilityFacade) {
        this.siteVersionAvailabilityFacade = siteVersionAvailabilityFacade;
    }

    public <T extends BaseBlock> Set<String> getAvailableVersionsLiterals(Class<T> blockClass) {
        return siteVersionAvailabilityFacade.getAvailabilityProvider(blockClass).getSiteVersions().stream()
                .map(SiteVersionType::getLiteral)
                .collect(Collectors.toSet());
    }

    public <T extends BaseBlock> Set<SiteVersionType> getAvailableVersions(Class<T> blockClass) {
        return siteVersionAvailabilityFacade.getAvailabilityProvider(blockClass).getSiteVersions();
    }

    public <T extends BaseBlock> Set<String> getBlockSiteVersions(BasePage page, Set<String> dependedSiteVersions,
                                            boolean turboDesktopAvailable, Class<T> blockClass) {
        var availableVersions = getAvailableVersionsLiterals(blockClass);
        var versions = availableVersions.stream()
                // filter base versions any block has
                .filter(it -> Set.of(DESKTOP.getLiteral(), MOBILE.getLiteral()).contains(it))
                .collect(Collectors.toSet());

        if (page instanceof PageWithAllowedAmp ampPage &&
                Boolean.TRUE.equals(ampPage.getAllowedAmp()) && availableVersions.contains(AMP.getLiteral())) {
            versions.add(AMP.getLiteral());
        }
        if (page instanceof PageWithAllowedTurbo turboPage && Boolean.TRUE.equals(turboPage.getAllowedTurbo())) {
            if (availableVersions.contains(TURBO.getLiteral())) {
                versions.add(TURBO.getLiteral());
            }
            if (availableVersions.contains(TURBO_DESKTOP.getLiteral()) && turboDesktopAvailable) {
                versions.add(TURBO_DESKTOP.getLiteral());
            }
        }
        dependedSiteVersions
                .stream().filter(it -> !it.equals(TURBO_DESKTOP.getLiteral()))
                .forEach(versions::add);

        return versions;
    }


    public Set<String> getSiteVersionsThatDependOnFeature(Class<? extends BaseBlock> modelClass, Set<String> features) {
        return siteVersionsThatDependOnFeature.getOrDefault(modelClass, Set.of())
                .stream()
                .filter(siteVersion -> features.contains(siteVersion + "_available"))
                .collect(Collectors.toSet());
    }
}
