package ru.yandex.partner.core.entity.block.type.strategy;

import java.math.BigDecimal;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithStrategy;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.core.operation.CoreModelProvider;
import ru.yandex.partner.core.props.CoreModel;
import ru.yandex.partner.core.props.ModelPropertyDefault;
import ru.yandex.partner.core.utils.CommonConverters;
import ru.yandex.partner.dbschema.partner.Tables;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.partner.core.CoreConstants.Strategies.MIN_CPM_STRATEGY_ID;
import static ru.yandex.partner.core.CoreConstants.Strategies.SEPARATE_CPM_STRATEGY_ID;
import static ru.yandex.partner.core.props.ChangePredicate.havingPropertyValue;

@Component
@ParametersAreNonnullByDefault
public class BlockWithStrategyRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupportWithMapper<BlockWithStrategy>
        implements CoreModelProvider<BlockWithStrategy>, BlockBkFiller<BlockWithStrategy> {

    //Несмотря на описанные зависимости от strategy_type, в апи отдаются все поля стратегии в списке редактируемых,
    //смотри: ApiEditableFieldsModelPart.kt; https://st.yandex-team.ru/PI-27538
    public static final CoreModel<BlockWithStrategy> MODEL = CoreModel.forClass(BlockWithStrategy.class)
            .property(ModelPropertyDefault.<BlockWithStrategy, Long>forProperty(BlockWithStrategy.STRATEGY_TYPE)
                    .withDefaultValueOnAdd(CoreConstants.Strategies.MAX_REVENUE_STRATEGY_ID)
            )
            .dependentPropertiesWhen(
                    havingPropertyValue(BlockWithStrategy.STRATEGY_TYPE, MIN_CPM_STRATEGY_ID),
                    configuration -> {
                        ModelPropertyDefault.forProperty(BlockWithStrategy.MINCPM)
                                .whenGhosted(BigDecimal.ZERO);
                        configuration.property(BlockWithStrategy.MINCPM);
                    }
            )
            .dependentPropertiesWhen(
                    havingPropertyValue(BlockWithStrategy.STRATEGY_TYPE, SEPARATE_CPM_STRATEGY_ID),
                    configuration -> {
                        configuration.property(BlockWithStrategy.MEDIA_ACTIVE);
                        configuration.property(BlockWithStrategy.MEDIA_CPM);
                        configuration.property(BlockWithStrategy.MEDIA_BLOCKED);
                        configuration.property(BlockWithStrategy.TEXT_ACTIVE);
                        configuration.property(BlockWithStrategy.TEXT_CPM);
                        configuration.property(BlockWithStrategy.TEXT_BLOCKED);
                    }
            )
            .build();


    private final JooqMapper<BlockWithStrategy> jooqMapper;

    public BlockWithStrategyRepositoryTypeSupport(DSLContext dslContext) {
        super(dslContext);
        this.jooqMapper = createBlockWithStrategyMapper();
    }

    private static JooqMapper<BlockWithStrategy> createBlockWithStrategyMapper() {
        return JooqMapperBuilder.<BlockWithStrategy>builder()
                .map(property(BlockWithStrategy.STRATEGY_TYPE, Tables.CONTEXT_ON_SITE_RTB.STRATEGY))
                .map(property(BlockWithStrategy.MINCPM, Tables.CONTEXT_ON_SITE_RTB.MINCPM))
                .map(convertibleProperty(BlockWithStrategy.MEDIA_ACTIVE, Tables.CONTEXT_ON_SITE_RTB.MEDIA_ACTIVE,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(convertibleProperty(BlockWithStrategy.MEDIA_BLOCKED, Tables.CONTEXT_ON_SITE_RTB.MEDIA_BLOCKED,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(property(BlockWithStrategy.MEDIA_CPM, Tables.CONTEXT_ON_SITE_RTB.MEDIA_CPM))
                .map(convertibleProperty(BlockWithStrategy.TEXT_ACTIVE, Tables.CONTEXT_ON_SITE_RTB.TEXT_ACTIVE,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(convertibleProperty(BlockWithStrategy.TEXT_BLOCKED, Tables.CONTEXT_ON_SITE_RTB.TEXT_BLOCKED,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(property(BlockWithStrategy.TEXT_CPM, Tables.CONTEXT_ON_SITE_RTB.TEXT_CPM))
                .map(convertibleProperty(BlockWithStrategy.VIDEO_ACTIVE, Tables.CONTEXT_ON_SITE_RTB.VIDEO_ACTIVE,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(convertibleProperty(BlockWithStrategy.VIDEO_BLOCKED, Tables.CONTEXT_ON_SITE_RTB.VIDEO_BLOCKED,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(property(BlockWithStrategy.VIDEO_CPM, Tables.CONTEXT_ON_SITE_RTB.VIDEO_CPM))
                .build();
    }

    @Override
    public JooqMapper<BlockWithStrategy> getJooqMapper() {
        return jooqMapper;
    }

    @Override
    public Class<BlockWithStrategy> getTypeClass() {
        return BlockWithStrategy.class;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(BlockWithStrategy model) {
        return MODEL.resolveEditableProperties(model);
    }

    @Override
    public CoreModel<BlockWithStrategy> getCoreModel() {
        return MODEL;
    }

    @Override
    public void fillBkData(@NotNull BlockWithStrategy model, @NotNull Page.TPartnerPage.TBlock.Builder bkData,
                           BlockBkDictContainer container) {
        Long strategyType = model.getStrategyType();
        bkData.setOptimizeType(SEPARATE_CPM_STRATEGY_ID.equals(strategyType) ? 0 : strategyType.intValue());
    }
}
