package ru.yandex.partner.core.entity.block.type.tags;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import NPartner.Page.TPartnerPage;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.JsonReaderWriterBuilders;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockBkDictContainer;
import ru.yandex.partner.core.entity.block.model.BlockWithTags;
import ru.yandex.partner.core.entity.block.repository.BlockBkFiller;
import ru.yandex.partner.core.entity.block.repository.type.AbstractBlockRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.entity.custombkoptions.CustomBkOptionsTypedRepository;
import ru.yandex.partner.core.entity.custombkoptions.filter.CustomBkOptionsFilters;
import ru.yandex.partner.core.entity.custombkoptions.model.CustomBkOptions;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.multistate.custombkoptions.CustomBkOptionsStateFlag;
import ru.yandex.partner.core.multitype.repository.EmptyTypeSupportModifyStrategy;
import ru.yandex.partner.core.operation.CoreModelProvider;
import ru.yandex.partner.core.props.CoreModel;
import ru.yandex.partner.core.props.ModelPropertyDefault;
import ru.yandex.partner.core.utils.CommonConverters;
import ru.yandex.partner.dbschema.partner.tables.ContextOnSiteRtb;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Component
@ParametersAreNonnullByDefault
public class BlockWithTagsRepositoryTypeSupport
        extends AbstractBlockRepositoryTypeSupportWithMapper<BlockWithTags>
        implements CoreModelProvider<BlockWithTags>, BlockBkFiller<BlockWithTags> {

    public static final CoreModel<BlockWithTags> MODEL = CoreModel.forClass(BlockWithTags.class)
            .property(ModelPropertyDefault.<BlockWithTags, List<Long>>forProperty(BlockWithTags.ORDER_TAGS)
                    .withDefaultValueOnAdd(List.of()))
            .property(ModelPropertyDefault.<BlockWithTags, List<Long>>forProperty(BlockWithTags.TARGET_TAGS)
                    .withDefaultValueOnAdd(List.of()))
            .build();

    private final JooqMapper<BlockWithTags> mapper;
    private final CustomBkOptionsTypedRepository bkOptionsTypedRepository;
    private final TagService tagService;

    @Autowired
    protected BlockWithTagsRepositoryTypeSupport(DSLContext dslContext, ObjectMapper objectMapper,
                                                 CustomBkOptionsTypedRepository bkOptionsTypedRepository,
                                                 TagService tagService) {
        super(dslContext, new EmptyTypeSupportModifyStrategy<>());
        this.mapper = createCommonBlockMapper(objectMapper);
        this.bkOptionsTypedRepository = bkOptionsTypedRepository;
        this.tagService = tagService;
    }

    private static JooqMapper<BlockWithTags> createCommonBlockMapper(ObjectMapper objectMapper) {
        return JooqMapperBuilder.<BlockWithTags>builder()
                .map(JsonReaderWriterBuilders.convertibleJsonProperty(
                        BlockWithTags.ORDER_TAGS,
                        ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.OPTS,
                        "order_tags",
                        value -> CommonConverters.jsonNodeToList(objectMapper, value, Long.class),
                        value -> CommonConverters.listToJsonNode(objectMapper, value)
                ))
                .map(JsonReaderWriterBuilders.convertibleJsonProperty(
                        BlockWithTags.TARGET_TAGS,
                        ContextOnSiteRtb.CONTEXT_ON_SITE_RTB.OPTS,
                        "target_tags",
                        value -> CommonConverters.jsonNodeToList(objectMapper, value, Long.class),
                        value -> CommonConverters.listToJsonNode(objectMapper, value)
                ))
                .build();
    }

    @Override
    public JooqMapper<BlockWithTags> getJooqMapper() {
        return mapper;
    }

    @Override
    public Class<BlockWithTags> getTypeClass() {
        return BlockWithTags.class;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<BlockWithTags> blocks) {
    }

    @Override
    public CoreModel<BlockWithTags> getCoreModel() {
        return MODEL;
    }

    @Override
    public void fillBkData(@NotNull BlockWithTags block, @NotNull TPartnerPage.TBlock.Builder bkData,
                           @NotNull BlockBkDictContainer container) {
        Map<Long, TagDto> tags = tagService.getTags();
        setTargetTags(block, bkData, tags);
        setOrderTags(block, bkData, tags);
    }

    @Override
    public void fillContainer(@NotNull BlockBkDictContainer container, @NotNull List<? extends BlockWithTags> models) {
        List<CustomBkOptions> options = bkOptionsTypedRepository.getAll(QueryOpts.forClass(CustomBkOptions.class)
                .withProps(CustomBkOptions.BK_NAME, CustomBkOptions.BK_VALUE)
                .withFilter(CoreFilterNode.create(
                        CustomBkOptionsFilters.MULTISTATE,
                        FilterOperator.IN,
                        not(has(CustomBkOptionsStateFlag.DELETED)))
                )
        );

        container.withCustomBkOptions(options);
    }

    private void setTargetTags(BlockWithTags block, TPartnerPage.TBlock.Builder bkData,
                               Map<Long, TagDto> tags) {
        if (block.getTargetTags() == null) {
            return;
        }
        block.getTargetTags().stream()
                .map(tags::get)
                .filter(Objects::nonNull)
                .map(TagDto::getName)
                .forEach(bkData::addTargetTags);
    }

    private void setOrderTags(BlockWithTags block, TPartnerPage.TBlock.Builder bkData,
                              Map<Long, TagDto> tags) {
        if (block.getOrderTags() == null) {
            return;
        }
        block.getOrderTags().stream()
                .map(tags::get)
                .filter(Objects::nonNull)
                .map(TagDto::getName)
                .forEach(bkData::addOrderTags);
    }
}
