package ru.yandex.partner.core.entity.block.type.tags;

import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.configuration.YTProperties;
import ru.yandex.partner.core.service.yt.YTService;
import ru.yandex.partner.defaultconfiguration.CacheNames;
import ru.yandex.partner.libs.exceptions.JsonParseException;

@ParametersAreNonnullByDefault
@Service
public class TagServiceImpl implements TagService {
    private static final String YT_TABLE_NAME = "dict/target_tags";

    private final YTService ytService;
    private final ObjectMapper objectMapper;
    private final YTProperties ytProperties;
    private final ObjectProvider<TagService> tagServiceObjectProvider;

    @Autowired
    public TagServiceImpl(YTService ytService, ObjectMapper objectMapper, YTProperties ytProperties,
                          ObjectProvider<TagService> tagServiceObjectProvider) {
        this.ytService = ytService;
        this.objectMapper = objectMapper;
        this.ytProperties = ytProperties;
        this.tagServiceObjectProvider = tagServiceObjectProvider;
    }

    @Override
    public Set<Long> getTagIds() {
        return tagServiceObjectProvider.getObject().getTags().keySet();
    }

    @Cacheable(value = {CacheNames.YT_TAGS_CACHE}, unless = "#result == null || #result.isEmpty()")
    @Override
    public SortedMap<Long, TagDto> getTags() {
        var tablePath = String.format("%s/%s", ytProperties.getPrefix(), YT_TABLE_NAME);
        var yql = String.format("id, name, descr FROM [%s] LIMIT 1000", tablePath);

        Function<Iterator<JsonNode>, List<TagDto>> callback = nodeIterator -> StreamEx.of(nodeIterator).map(it -> {
            try {
                return objectMapper.treeToValue(it, TagDto.class);
            } catch (JsonProcessingException e) {
                throw new JsonParseException(e.getMessage());
            }
        }).collect(Collectors.toList());

        return new TreeMap<>(ytService.selectFromAnyReplicas(yql, callback).stream().collect(Collectors.toMap(
                TagDto::getId,
                Function.identity()
        )));
    }
}
