package ru.yandex.partner.core.entity.blockseq;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.TableField;
import org.jooq.impl.DSL;

import ru.yandex.partner.core.entity.blockseq.model.BlockSequence;
import ru.yandex.partner.core.entity.simplemodels.AbstractSimpleRepository;
import ru.yandex.partner.core.entity.utils.DSLUtils;
import ru.yandex.partner.core.filter.container.ModelFilterContainer;

public abstract class BlockSequenceRepository<BS extends BlockSequence, R extends Record>
        extends AbstractSimpleRepository<BS, R> {
    private final Map<String, Field<?>> dbFields;
    private final BlockSequenceFieldsProvider<BS, R> blockSequenceFieldsProvider;

    public BlockSequenceRepository(
            DSLContext dslContext,
            BlockSequenceFieldsProvider<BS, R> blockSequenceFieldsProvider,
            ModelFilterContainer<BS> modelFilterContainer
    ) {
        super(dslContext, modelFilterContainer);
        this.blockSequenceFieldsProvider = blockSequenceFieldsProvider;
        this.dbFields = prepareDbFieldsMapping();
    }

    public BS getBlockSeqInstance() {
        return blockSequenceFieldsProvider.getBlockSeqInstance();
    }

    /**
     * использовать в случе если есть уверенность,
     * что не будет одновременных вставок и апдейтов
     */
    @Override
    public List<BS> createOrUpdate(List<BS> entities) {
        var query = getDslContext().insertInto(
                getTable(),
                getPageIdField(),
                getNextBlockIdField()
        );
        for (var entity : entities) {
            query.values(
                    entity.getId(),
                    entity.getNextBlockId()
            );
        }

        query.onDuplicateKeyUpdate()
                .set(getNextBlockIdField(),
                        DSLUtils.getValuesStatement(getNextBlockIdField()));

        var queryResult = query.returning().fetch();
        return queryResult.map(r -> fetchQueryFunction().apply(r));
    }

    @Override
    public int insert(List<BS> entities) {
        var query = getDslContext().insertInto(
                getTable(),
                getPageIdField(),
                getNextBlockIdField()
        );
        for (var entity : entities) {
            query.values(
                    entity.getId(),
                    entity.getNextBlockId()
            );
        }
        return query.execute();
    }

    @Override
    public int update(List<BS> entities) {
        if (entities.isEmpty()) {
            return 0;
        }

        var caseOption = DSL.when(getPageIdField().eq(entities.get(0).getId()),
                entities.get(0).getNextBlockId());
        for (int i = 1; i < entities.size(); i++) {
            caseOption.when(getPageIdField().eq(entities.get(i).getId()),
                    entities.get(i).getNextBlockId());
        }
        var updateQuery = getDslContext().update(getTable())
                .set(getNextBlockIdField(), caseOption)
                .where(getPageIdField().in(
                        StreamEx.of(entities).map(BlockSequence::getId).toList()));
        return updateQuery.execute();
    }

    @Override
    public void delete(Collection<BS> currentModels) {
        throw new UnsupportedOperationException("Метод не реализован");
    }

    @Override
    protected Function<Record, BS> fetchQueryFunction() {
        return r -> {
            BS seqModel = getBlockSeqInstance();
            seqModel.setId(r.get(getPageIdField()));
            seqModel.setNextBlockId(r.get(getNextBlockIdField()));
            return seqModel;
        };
    }

    @Override
    protected Map<String, Field<?>> getDbFields() {
        return this.dbFields;
    }

    @Override
    protected Field<?> getIdField() {
        return getPageIdField();
    }


    private Map<String, Field<?>> prepareDbFieldsMapping() {
        return Map.of(
                BlockSequence.ID.name(), getPageIdField(),
                BlockSequence.NEXT_BLOCK_ID.name(), getNextBlockIdField());
    }

    public TableField<R, Long> getPageIdField() {
        return blockSequenceFieldsProvider.getPageIdField();
    }

    private TableField<R, Long> getNextBlockIdField() {
        return blockSequenceFieldsProvider.getNextBlockIdField();
    }
}
