package ru.yandex.partner.core.entity.blockseq;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;
import org.jooq.Record;

import ru.yandex.partner.core.block.BlockUniqueIdConverter;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.blockseq.filter.BlockSequenceFilters;
import ru.yandex.partner.core.entity.blockseq.model.BlockSequence;
import ru.yandex.partner.core.entity.utils.TransactionChecker;
import ru.yandex.partner.core.filter.CoreFilterNode;

import static ru.yandex.partner.core.CoreConstants.TECHNICAL_RTB_BLOCK_ID;

public final class SimpleBlockSequenceService<BS extends BlockSequence, R extends Record>
        implements BlockSequenceService {
    private final BlockSequenceRepository<BS, R> repository;
    private final BlockUniqueIdConverter.Prefixes prefix;
    private final Class<? extends BaseBlock> blockClass;

    public SimpleBlockSequenceService(
            BlockSequenceRepository<BS, R> repository,
            BlockUniqueIdConverter.Prefixes prefix,
            Class<? extends BaseBlock> blockClass
    ) {
        this.repository = repository;
        this.prefix = prefix;
        this.blockClass = blockClass;
    }

    @Override
    public Map<Long, List<Long>> getNextBlockIds(Map<Long, Long> pageIdsWithBlockCount, boolean dryRun) {
        if (!dryRun) {
            //если не в транзакции - Exception
            TransactionChecker.requiredTransaction();
        }

        var pageIds = pageIdsWithBlockCount.keySet();

        var blockSequencesToUpdate = repository.getAll(
                        CoreFilterNode.in(BlockSequenceFilters.PAGE_ID, pageIds),
                        null,
                        null,
                        !dryRun
                ).stream()
                .collect(Collectors.toMap(BlockSequence::getId, Function.identity()));

        var blockSequencesToAdd = StreamEx.of(pageIds)
                .filter(it -> !blockSequencesToUpdate.containsKey(it))
                .map(it -> {
                    var blockSeq = repository.getBlockSeqInstance();
                    blockSeq.setId(it);
                    blockSeq.setNextBlockId(1L);
                    return blockSeq;
                })
                .toMap(BlockSequence::getId, Function.identity());

        Map<Long, List<Long>> result = Maps.newHashMapWithExpectedSize(pageIds.size());
        pageIdsWithBlockCount.forEach((pageId, blockCount) -> {
            var blockSeq = blockSequencesToUpdate.getOrDefault(pageId, blockSequencesToAdd.get(pageId));
            var cur = blockSeq.getNextBlockId();
            var list = Lists.<Long>newArrayListWithExpectedSize(Math.toIntExact(blockCount));
            for (int i = 0; i < blockCount; i++) {
                list.add(cur);
                cur = incrementBlockId(cur);
            }
            result.put(pageId, list);
            blockSeq.setNextBlockId(cur);
        });

        if (!dryRun) {
            repository.update(StreamEx.of(blockSequencesToUpdate.values()).toList());
            repository.insert(StreamEx.of(blockSequencesToAdd.values()).toList());
        }

        return result;
    }

    @Override
    public Class<? extends BaseBlock> getBlockClass() {
        return blockClass;
    }

    @Override
    public BlockUniqueIdConverter.Prefixes getUniqueIdPrefix() {
        return prefix;
    }


    private long incrementBlockId(long id) {
        id += 1;
        if (id == TECHNICAL_RTB_BLOCK_ID) {
            id += 1;
        }
        return id;
    }

}
