package ru.yandex.partner.core.entity.brand;

import java.util.List;
import java.util.Map;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.TableImpl;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.brand.filter.BrandModelFilterContainer;
import ru.yandex.partner.core.entity.simplemodels.AbstractSimpleRepositoryWithCommonMethods;
import ru.yandex.partner.core.utils.CommonConverters;
import ru.yandex.partner.dbschema.partner.tables.Brands;
import ru.yandex.partner.dbschema.partner.tables.records.BrandsRecord;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.partner.dbschema.partner.tables.Brands.BRANDS;

@Repository
@ParametersAreNonnullByDefault
public class BrandRepository extends AbstractSimpleRepositoryWithCommonMethods<Brand, BrandsRecord> {

    private final Map<String, Field<?>> dbFields;

    public BrandRepository(DSLContext dslContext, BrandModelFilterContainer brandModelFilterContainer) {
        super(dslContext,
                brandModelFilterContainer,
                createCommonMapper(),
                List.of(Brands.BRANDS.BID,
                        Brands.BRANDS.BLOCK_ID,
                        Brands.BRANDS.PAGE_ID));
        this.dbFields = prepareDbFieldsMapping();
    }


    private static JooqMapper<Brand> createCommonMapper() {
        return JooqMapperBuilder.<Brand>builder()
                .map(property(Brand.BID, BRANDS.BID))
                .map(property(Brand.BLOCK_ID, BRANDS.BLOCK_ID))
                .map(property(Brand.CPM, BRANDS.CPM))
                .map(convertibleProperty(Brand.BLOCKED, BRANDS.BLOCKED,
                        CommonConverters::booleanFromLong, CommonConverters::booleanToLong))
                .map(property(Brand.PAGE_ID, BRANDS.PAGE_ID))
                .build();
    }

    private Map<String, Field<?>> prepareDbFieldsMapping() {
        return Map.of(
                Brand.ID.name(), BRANDS.ID,
                Brand.BID.name(), BRANDS.BID,
                Brand.BLOCK_ID.name(), BRANDS.BLOCK_ID,
                Brand.CPM.name(), BRANDS.CPM,
                Brand.BLOCKED.name(), BRANDS.BLOCKED,
                Brand.PAGE_ID.name(), BRANDS.PAGE_ID
        );
    }

    @Override
    protected TableImpl<BrandsRecord> getTable() {
        return BRANDS;
    }

    @Override
    protected Function<Record, Brand> fetchQueryFunction() {
        return r -> getMapper().fromDb(r, new Brand());
    }

    @Override
    protected Map<String, Field<?>> getDbFields() {
        return this.dbFields;
    }

    @Override
    protected Field<?> getIdField() {
        return BRANDS.ID;
    }

    @Override
    protected Class<Brand> getModelClass() {
        return Brand.class;
    }
}
