package ru.yandex.partner.core.entity.dictionary.moderationreason;

import java.util.Collection;
import java.util.Map;

import org.jooq.DSLContext;
import org.jooq.SelectConditionStep;
import org.springframework.stereotype.Repository;

import ru.yandex.partner.core.dictionary.moderationreason.ModerationReason;
import ru.yandex.partner.dbschema.partner.tables.records.ModerationReasonRecord;

import static org.jooq.impl.DSL.bitAnd;
import static ru.yandex.partner.core.DbLocale.EN;
import static ru.yandex.partner.core.DbLocale.RU;
import static ru.yandex.partner.dbschema.partner.tables.ModerationReason.MODERATION_REASON;

@Repository
public class ModerationReasonRepository {

    private static final long DELETED_STATE_VALUE = 1L;

    /**
     * Все причины модерации
     *
     * @param dslContext
     * @return
     */
    public Map<Long, ModerationReason> getAllModerationReasons(DSLContext dslContext) {
        return getModerationReasonsInternal(dslContext, null);
    }

    /**
     * Причины модерации по переданному списку id
     *
     * @param dslContext
     * @param moderationReasonIds
     * @return
     */
    public Map<Long, ModerationReason> getModerationReasons(DSLContext dslContext,
                                                            Collection<Long> moderationReasonIds) {
        if (moderationReasonIds == null || moderationReasonIds.isEmpty()) {
            return Map.of();
        }

        return getModerationReasonsInternal(dslContext, moderationReasonIds);
    }

    private Map<Long, ModerationReason> getModerationReasonsInternal(DSLContext dslContext,
                                                                     Collection<Long> moderationReasonIds) {
        SelectConditionStep<ModerationReasonRecord> query = dslContext
                .selectFrom(MODERATION_REASON)
                .where(bitAnd(DELETED_STATE_VALUE, MODERATION_REASON.MULTISTATE).eq(0L));

        if (moderationReasonIds != null) {
            query = query.and(MODERATION_REASON.ID.in(moderationReasonIds));
        }

        return query.fetchMap(MODERATION_REASON.ID, ModerationReasonRepository::recordToEntity);
    }

    private static ModerationReason recordToEntity(ModerationReasonRecord moderationReasonRecord) {
        return new ModerationReason(
                moderationReasonRecord.getId(),
                Map.of(
                        RU, moderationReasonRecord.getManagerTxtRu(),
                        EN, moderationReasonRecord.getManagerTxtEn()
                ),
                Map.of(
                        RU, moderationReasonRecord.getPartnerTxtRu(),
                        EN, moderationReasonRecord.getPartnerTxtEn()
                )
        );
    }

}
