package ru.yandex.partner.core.entity.dictionary.moderationreason;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.DbLocale;
import ru.yandex.partner.core.dictionary.moderationreason.ModerationReason;

@Service
public class ModerationReasonService {

    private final DSLContext dslContext;
    private final ModerationReasonRepository moderationReasonRepository;

    @Autowired
    public ModerationReasonService(DSLContext dslContext, ModerationReasonRepository moderationReasonRepository) {
        this.dslContext = dslContext;
        this.moderationReasonRepository = moderationReasonRepository;
    }

    public Map<Long, String> getDictionaryValues(String language, boolean isManager) {
        Map<Long, ModerationReason> moderationReasons = moderationReasonRepository.getAllModerationReasons(dslContext);
        return render(moderationReasons, language, isManager);
    }

    public <K> Map<K, String> renderModerationReason(Map<K, List<Long>> moderationReasonIdsMap,
                                                     String language, boolean isManager) {

        Set<Long> moderationReasonIds = moderationReasonIdsMap.values().stream()
                .flatMap(Collection::stream).collect(Collectors.toSet());

        Map<Long, ModerationReason> moderationReasons = moderationReasonRepository.getModerationReasons(dslContext,
                moderationReasonIds);

        Map<Long, String> textMap = render(moderationReasons, language, isManager);

        return moderationReasonIdsMap.entrySet().stream()
                .collect(Collectors.toMap(Entry::getKey, entry -> getTextForReasons(textMap, entry.getValue())));

    }

    @Nonnull
    private Map<Long, String> render(Map<Long, ModerationReason> moderationReasons,
                                     String language, boolean isManager) {
        DbLocale dbLocale = DbLocale.forLanguage(language);

        return moderationReasons.entrySet().stream().collect(Collectors.toMap(
                Entry::getKey,
                entry -> getTextForReason(entry.getValue(), isManager, dbLocale)
        ));
    }

    private String getTextForReasons(Map<Long, String> textMap, List<Long> reasonIds) {
        return reasonIds.stream()
                .map(textMap::get)
                .filter(Objects::nonNull)
                .distinct()
                .sorted()
                .collect(Collectors.joining("\n"));
    }

    private String getTextForReason(ModerationReason moderationReason, boolean isManager, DbLocale dbLocale) {
        return isManager
                ? moderationReason.getManagerTxt(dbLocale)
                : moderationReason.getPartnerTxt(dbLocale);

    }

}
