package ru.yandex.partner.core.entity.dsp;

import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.block.model.BlockWithDspsDspsUnmoderatedDspModeStrategy;
import ru.yandex.partner.core.entity.dsp.model.BaseDsp;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.model.DspWithCommonFields;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static ru.yandex.partner.core.messages.DspCaptionMsg.MEDIA_ADVERTISING;
import static ru.yandex.partner.core.messages.DspCaptionMsg.MOBILE_ADVERTISING;
import static ru.yandex.partner.core.messages.DspCaptionMsg.VIDEO_ADVERTISING;

public class DspConstants {
    public static final Set<ModelProperty<? super Dsp, ?>> EDIT_FORBIDDEN_MODEL_PROPERTIES = Set.of(
            BaseDsp.ID,
            BaseDsp.PUBLIC_ID,
            DspWithCommonFields.IS_DELETED,
            DspWithCommonFields.IS_CREATED_IN_BK
    );

    public static final Map<Long, GettextMsg> DSP_TYPE_CAPTIONS = Map.of(
            0L, MEDIA_ADVERTISING,
            1L, VIDEO_ADVERTISING,
            2L, MOBILE_ADVERTISING
    );

    public enum DspMode {
        AUTO("auto"),
        WHITELIST("whitelist"),
        //        BLACKLIST("blacklist"),
        FORCE("force");

        static SortedMap<String, DspMode> dspModeMap = new TreeMap<>(Arrays.stream(DspMode.values())
                .collect(Collectors.toMap(m -> m.literal, m -> m)));
        private final String literal;

        DspMode(String l) {
            this.literal = l;
        }

        public static DspMode fromBlock(@NotNull BlockWithDspsDspsUnmoderatedDspModeStrategy block) {
            return Optional.of(block)
                    .map(BlockWithDspsDspsUnmoderatedDspModeStrategy::getDspMode)
                    .map(m -> dspModeMap.get(m))
                    .orElseThrow(() -> new IllegalArgumentException(
                            "Can't extract dsp_mode from value: %s".formatted(block.getDspMode()))
                    );
        }

        public String getLiteral() {
            return literal;
        }

        public String getBkLiteral() {
            return literal.toUpperCase();
        }

        public static boolean dspModeExists(String dspMode) {
            return dspModeMap.containsKey(dspMode);
        }

        public static Set<String> dspModeValues() {
            return dspModeMap.keySet();
        }
    }

    private DspConstants() {
    }
}
