package ru.yandex.partner.core.entity.dsp.multistate;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import io.micrometer.core.lang.NonNullApi;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.libs.multistate.action.ActionCheck;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;

import static ru.yandex.partner.core.CoreConstants.DSP_TYPES;

@Service
@NonNullApi
public class DspActionChecksService {
    private final Map<DspActionCheck, ActionCheck<Dsp>> checksMap = Map.of(
            DspActionCheck.IS_ALLOW_DSP_VIA_LOGBROKER, new ActionCheck<Dsp>(this::checkIsAllowDspViaLogbroker),
            DspActionCheck.HAS_ALL_TYPES, new ActionCheck<Dsp>(this::checkHasAllTypes, Set.of(Dsp.TYPES)),
            DspActionCheck.HAS_YANDEX_TYPE, new ActionCheck<Dsp>(this::checkHasYandexType, Set.of(Dsp.TYPES))
    );

    public DspActionChecksService() {
    }

    public Function<List<? extends Dsp>, List<Boolean>> getCheckFunction(DspActionCheck check) {
        return checksMap.get(check).getCheck();
    }

    public List<Boolean> checkIsAllowDspViaLogbroker(List<? extends Dsp> models) {
        // По факту в перловых конфигах выставлено всегда 1
        return models.stream().map((m) -> true).collect(Collectors.toList());
    }

    public List<Boolean> checkHasAllTypes(List<? extends Dsp> models) {
        return models.stream()
                .map(Dsp::getTypes)
                .map(this::hasAllTypes)
                .collect(Collectors.toList());
    }

    public Boolean hasAllTypes(List<Long> types) {
        if (types == null || types.isEmpty()) {
            return false;
        }
        Collections.sort(types);
        return DSP_TYPES.equals(types);
    }

    public List<Boolean> checkHasYandexType(List<? extends Dsp> models) {
        return models.stream()
                .map(Dsp::getTypes)
                .map(t -> t.contains(0L))
                .collect(Collectors.toList());
    }

    public enum DspActionCheck implements ActionCheckId {
        IS_ALLOW_DSP_VIA_LOGBROKER,
        HAS_ALL_TYPES,
        HAS_YANDEX_TYPE
    }

}
