package ru.yandex.partner.core.entity.dsp.multistate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.dsp.actions.DspActionsEnum;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.messages.DspActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.dsp.DspMultistate;
import ru.yandex.partner.core.multistate.dsp.DspStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.CREATED_IN_BK;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.CREATED_IN_PI;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.EDITED;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.LINKED_IN_BALANCE;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.NEED_APPROVE;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.NEED_CREATE_IN_PI;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.NOT_NEED_CREATE_IN_PI;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.WORKING;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.WORKING_ON_ALL_PLATFORMS;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.WORKING_ON_YANDEX_SERVICES;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Component
public class DspMultistateGraph extends AbstractMultistateGraph<Dsp, DspStateFlag> {

    private final DspActionChecksService dspActionChecksService;

    public DspMultistateGraph(DspActionChecksService dspActionChecksService) {
        super();
        this.dspActionChecksService = dspActionChecksService;
    }

    @Override
    public DspMultistate getMultistateFromModel(Dsp model) {
        return model.getMultistate();
    }

    @Override
    protected Multistate<DspStateFlag> getMultistateForValue(Long multstateValue) {
        return new DspMultistate(multstateValue);
    }

    @Override
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends Dsp> models) {
        if (!(check instanceof DspActionChecksService.DspActionCheck)) {
            return super.performCheck(check, models);
        }
        Function<List<? extends Dsp>, List<Boolean>> checkFunction =
                dspActionChecksService.getCheckFunction((DspActionChecksService.DspActionCheck) check);
        return checkFunction == null ? super.performCheck(check, models)
                : checkFunction.apply(models);
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<Dsp, DspStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<Dsp, DspStateFlag>> actionEntryMap = new HashMap<>();

        actionEntryMap.put(DspActionsEnum.CREATE_IN_BK,
                getActionEntryBuilder(DspActionMsg.CREATE_IN_BK)
                        .setPredicate(not(has(CREATED_IN_BK)).and(not(has(NEED_APPROVE))).and(not(has(DELETED))))
                        .setSetFlags(CREATED_IN_BK, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.CREATE_IN_PI,
                getActionEntryBuilder(DspActionMsg.CREATE_IN_PI)
                        .setPredicate(has(NEED_CREATE_IN_PI))
                        .setSetFlags(Map.of(
                                CREATED_IN_PI, true,
                                NEED_CREATE_IN_PI, false
                        ))
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.LINK_IN_BALANCE,
                getActionEntryBuilder(DspActionMsg.LINK_IN_BALANCE)
                        .setPredicate(not(has(LINKED_IN_BALANCE)).and(not(has(NEED_APPROVE))).and(not(has(DELETED))))
                        .setSetFlags(LINKED_IN_BALANCE, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.START,
                getActionEntryBuilder(DspActionMsg.START)
                        .setPredicate(
                                has(CREATED_IN_BK).and(has(CREATED_IN_PI).or(has(NOT_NEED_CREATE_IN_PI)))
                                        .and(has(LINKED_IN_BALANCE)).and(not(has(WORKING))).and(not(has(DELETED))))
                        .setSetFlags(WORKING, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.STOP,
                getActionEntryBuilder(DspActionMsg.STOP)
                        .setPredicate(has(WORKING))
                        .setSetFlags(WORKING, false)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.REQUEST_FOR_APPROVAL,
                getActionEntryBuilder(DspActionMsg.REQUEST_FOR_APPROVAL)
                        .setPredicate(empty())
                        .setSetFlags(NEED_APPROVE, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.APPROVE,
                getActionEntryBuilder(DspActionMsg.APPROVE)
                        .setPredicate(has(NEED_APPROVE).and(not(has(DELETED))))
                        .setSetFlags(NEED_APPROVE, false)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.EDIT,
                getActionEntryBuilder(DspActionMsg.EDIT)
                        .setPredicate(not(has(NEED_APPROVE)).and(not(has(DELETED))))
                        .setSetFlags(EDITED, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.START_ON_ALL_PLATFORMS,
                getActionEntryBuilder(DspActionMsg.START_ON_ALL_PLATFORMS)
                        .setPredicate(
                                has(CREATED_IN_BK)
                                        .and(has(CREATED_IN_PI).or(has(NOT_NEED_CREATE_IN_PI)))
                                        .and(has(LINKED_IN_BALANCE))
                                        .and(not(has(WORKING_ON_ALL_PLATFORMS))).and(not(has(DELETED))))
                        .setSetFlags(WORKING_ON_ALL_PLATFORMS, true)
                        .setChecks(List.of(DspActionChecksService.DspActionCheck.HAS_ALL_TYPES))
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.STOP_ON_ALL_PLATFORMS,
                getActionEntryBuilder(DspActionMsg.STOP_ON_ALL_PLATFORMS)
                        .setPredicate(has(WORKING_ON_ALL_PLATFORMS))
                        .setSetFlags(WORKING_ON_ALL_PLATFORMS, false)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.START_ON_YANDEX_SERVICES,
                getActionEntryBuilder(DspActionMsg.START_ON_YANDEX_SERVICES)
                        .setPredicate(
                                has(CREATED_IN_BK)
                                        .and(has(CREATED_IN_PI))
                                        .and(has(LINKED_IN_BALANCE))
                                        .and(not(has(WORKING_ON_YANDEX_SERVICES)))
                                        .and(not(has(DELETED))))
                        .setSetFlags(WORKING_ON_YANDEX_SERVICES, true)
                        .setChecks(List.of(DspActionChecksService.DspActionCheck.HAS_YANDEX_TYPE))
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.STOP_ON_YANDEX_SERVICES,
                getActionEntryBuilder(DspActionMsg.STOP_ON_YANDEX_SERVICES)
                        .setPredicate(has(WORKING_ON_YANDEX_SERVICES))
                        .setSetFlags(WORKING_ON_YANDEX_SERVICES, false)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.TEST,
                getActionEntryBuilder(DspActionMsg.TEST)
                        .setPredicate(has(CREATED_IN_BK).and(not(has(DELETED))))
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.ADD,
                getActionEntryBuilder(DspActionMsg.ADD)
                        .setPredicate(empty())
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.DELETE,
                getActionEntryBuilder(DspActionMsg.DELETE)
                        .setPredicate(
                                not(has(DELETED))
                                        .and(not(has(WORKING)))
                                        .and(not(has(WORKING_ON_ALL_PLATFORMS)))
                                        .and(not(has(WORKING_ON_YANDEX_SERVICES)))
                        )
                        .setSetFlags(DELETED, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.RESTORE,
                getActionEntryBuilder(DspActionMsg.RESTORE)
                        .setPredicate(has(DELETED))
                        .setSetFlags(DELETED, false)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.SET_FLAG_NEED_CREATE_IN_PI,
                getActionEntryBuilder(DspActionMsg.SET_FLAG_NEED_CREATE_IN_PI)
                        .setPredicate(
                                not(has(NEED_CREATE_IN_PI))
                                        .and(not(has(NEED_APPROVE)))
                                        .and(not(has(DELETED)))
                                        .and(not(has(CREATED_IN_PI)))
                        )
                        .setSetFlags(NEED_CREATE_IN_PI, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.SET_FLAG_NOT_NEED_CREATE_IN_PI,
                getActionEntryBuilder(DspActionMsg.SET_FLAG_NOT_NEED_CREATE_IN_PI)
                        .setPredicate(
                                not(has(NOT_NEED_CREATE_IN_PI))
                                        .and(not(has(NEED_APPROVE)))
                                        .and(not(has(DELETED)))
                                        .and(not(has(CREATED_IN_PI)))
                        )
                        .setSetFlags(NOT_NEED_CREATE_IN_PI, true)
                        .build()
        );
        actionEntryMap.put(DspActionsEnum.RESEND_TO_BK,
                getActionEntryBuilder(DspActionMsg.RESEND_TO_BK)
                        .setPredicate(has(CREATED_IN_BK))
                        .setChecks(List.of(DspActionChecksService.DspActionCheck.IS_ALLOW_DSP_VIA_LOGBROKER))
                        .build()
        );

        return actionEntryMap;
    }

    private ActionEntry.Builder<Dsp, DspStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(Dsp.ID, Dsp.MULTISTATE));
    }

    @Override
    public Class<Dsp> getModelClass() {
        return Dsp.class;
    }

    @Override
    public Multistate<DspStateFlag> convertMultistate(List<DspStateFlag> enabledFlags) {
        return new DspMultistate(enabledFlags);
    }
}
