package ru.yandex.partner.core.entity.dsp.repository;

import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Table;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.repository.filter.ConditionFilter;
import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.dsp.container.DspRepositoryContainer;
import ru.yandex.partner.core.entity.dsp.filter.DspFilters;
import ru.yandex.partner.core.entity.dsp.filter.DspModelFilterContainer;
import ru.yandex.partner.core.entity.dsp.model.BaseDsp;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.repository.type.DspRepositoryTypeSupportFacade;
import ru.yandex.partner.core.entity.utils.ConditionUtils;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.multitype.repository.PartnerTypedRepository;
import ru.yandex.partner.dbschema.partner.Tables;

import static ru.yandex.partner.core.entity.dsp.repository.filter.DspFilterFactory.dspIdFilter;
import static ru.yandex.partner.dbschema.partner.tables.BlockDsps.BLOCK_DSPS;
import static ru.yandex.partner.dbschema.partner.tables.BlockDspsUnmoderated.BLOCK_DSPS_UNMODERATED;

@Repository
@ParametersAreNonnullByDefault
public class DspTypedRepository extends PartnerTypedRepository<BaseDsp, Long, DspRepositoryContainer,
        DspRepositoryContainer> {

    @Autowired
    public DspTypedRepository(DspRepositoryTypeSupportFacade typeSupportFacade, DSLContext dslContext,
                              DspModelFilterContainer metaFilterService) {
        super(dslContext, typeSupportFacade, metaFilterService);
    }

    @Override
    protected ConditionFilter getIdFilter(Collection<Long> dspIds) {
        return dspIdFilter(dspIds);
    }

    @Override
    protected Table<?> getBaseTable() {
        return Tables.DSP;
    }

    public BaseDsp getDspById(Long dspId) {
        List<BaseDsp> models = getAll(QueryOpts.forClass(Dsp.class).withFilterByIds(Set.of(dspId)));
        if (models != null && !models.isEmpty()) {
            return models.get(0);
        }
        return null;
    }

    public Map<PageBlockIds, List<Dsp>> getDspsForPageBlockIds(
            DSLContext dslContext,
            Collection<PageBlockIds> pageBlockIdsList,
            @Nullable Set<ModelProperty<? extends Model, ?>> props
    ) {
        if (pageBlockIdsList.isEmpty()) {
            return Map.of();
        }

        Map<PageBlockIds, List<Long>> pageBlockIdsListMap = dslContext
                .select(
                        BLOCK_DSPS.PAGE_ID,
                        BLOCK_DSPS.BLOCK_ID,
                        BLOCK_DSPS.DSP_ID
                )
                .from(BLOCK_DSPS)
                .where(
                        ConditionUtils.toPageBlockCondition(pageBlockIdsList, BLOCK_DSPS.PAGE_ID, BLOCK_DSPS.BLOCK_ID)
                                .and(BLOCK_DSPS.IS_DELETED.eq(0L))
                ).fetchGroups(
                        record -> new PageBlockIds(record.get(BLOCK_DSPS.PAGE_ID), record.get(BLOCK_DSPS.BLOCK_ID)),
                        record -> record.get(BLOCK_DSPS.DSP_ID)
                );

        List<Long> dspIds = StreamEx.ofValues(pageBlockIdsListMap)
                .flatMap(Collection::stream).distinct().collect(Collectors.toList());

        var dsps = getAll(QueryOpts.forClass(Dsp.class)
                .withFilterByIds(dspIds)
                .withProps(props)
        );

        Map<Long, Dsp> dspByIds = StreamEx.of(dsps)
                .map(Dsp.class::cast)
                .toMap(Dsp::getId, d -> d);

        return EntryStream.of(pageBlockIdsListMap).mapValues(
                v -> v.stream().map(dspByIds::get).sorted(Comparator.comparing(Dsp::getShortCaption))
                        .collect(Collectors.toList())
        ).toMap();
    }

    public Map<PageBlockIds, List<Long>> getDspsUnmoderatedForPageBlockIds(DSLContext dslContext,
                                                                           Collection<PageBlockIds> pageBlockIdsList) {
        if (pageBlockIdsList.isEmpty()) {
            return Map.of();
        }

        return dslContext
                .select(
                        BLOCK_DSPS_UNMODERATED.PAGE_ID,
                        BLOCK_DSPS_UNMODERATED.BLOCK_ID,
                        BLOCK_DSPS_UNMODERATED.DSP_ID
                )
                .from(BLOCK_DSPS_UNMODERATED)
                .where(
                        ConditionUtils.toPageBlockCondition(pageBlockIdsList,
                                BLOCK_DSPS_UNMODERATED.PAGE_ID, BLOCK_DSPS_UNMODERATED.BLOCK_ID)
                )
                .fetchGroups(
                        record -> new PageBlockIds(
                                record.get(BLOCK_DSPS_UNMODERATED.PAGE_ID), record.get(BLOCK_DSPS_UNMODERATED.BLOCK_ID)
                        ),
                        record -> record.get(BLOCK_DSPS_UNMODERATED.DSP_ID)
                );
    }


    @Override
    protected <S extends BaseDsp> CoreFilterNode<S> getCoreFilterNodeById(Collection<Long> ids) {
        return DspFilters.ID.in(ids);
    }

    @Override
    protected ModelProperty<? extends Model, Long> getIdModelProperty() {
        return BaseDsp.ID;
    }
}
