package ru.yandex.partner.core.entity.dsp.rules;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import one.util.streamex.StreamEx;

import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithBlockType;
import ru.yandex.partner.core.entity.block.model.InternalRtbBlock;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.dsp.model.Dsp;

import static ru.yandex.partner.core.CoreConstants.VIDEO_BANNER_DSPS;
import static ru.yandex.partner.core.CoreConstants.VIDEO_INTERSTITIAL_DSPS;
import static ru.yandex.partner.core.block.MobileBlockType.BANNER;
import static ru.yandex.partner.core.block.MobileBlockType.INTERSTITIAL;
import static ru.yandex.partner.core.block.MobileBlockType.REWARDED;

public class DspRuleContainer {
    private static final TreeSet<Long> RTB_VIDEO_DSPS =
            StreamEx.of(CoreConstants.DSP_DIRECT_ID)
                    .append(CoreConstants.EXTERNAL_CONTEXT_ON_SITE_RTB_DEFAULT_VIDEO_DSPS)
                    .toSetAndThen(TreeSet::new);

    private static final TreeSet<Long> INTERNAL_RTB_VIDEO_DSPS =
            StreamEx.of(CoreConstants.DSP_DIRECT_ID)
                    .append(CoreConstants.INTERNAL_CONTEXT_ON_SITE_RTB_DEFAULT_VIDEO_DSPS)
                    .toSetAndThen(TreeSet::new);

    /**
     * Список video-dsp для валидации
     */
    private final Map<Class<? extends BaseBlock>, Set<Long>> videoDspsForValidationByModel;
    /**
     * Список dsp на добавление/удаление при изменении show_video (исключён direct)
     * в разрезе типов блоков.
     */
    private final Map<Class<? extends BaseBlock>, Set<Long>> videoDspsByModel;
    /**
     * Список video-dsp для мобилок (для валидации или добавления/удаления при изменении show_video)
     */
    private final Map<String, Set<Long>> mobileVideoDspsMapByBlockType;

    private Map<Long, Dsp> dspMap;
    private Map<Long, List<Dsp>> blockAvailableDspsMap;

    private Set<Long> dspTypes = new TreeSet<>();

    private Boolean canEditRichMedia = false;

    public DspRuleContainer() {
        this.videoDspsForValidationByModel = Map.of(
                RtbBlock.class, RTB_VIDEO_DSPS,
                InternalRtbBlock.class, INTERNAL_RTB_VIDEO_DSPS
        );

        this.videoDspsByModel = Map.of(
                RtbBlock.class, CoreConstants.EXTERNAL_CONTEXT_ON_SITE_RTB_DEFAULT_VIDEO_DSPS,
                InternalRtbBlock.class, CoreConstants.INTERNAL_CONTEXT_ON_SITE_RTB_DEFAULT_VIDEO_DSPS
        );

        this.mobileVideoDspsMapByBlockType = Map.of(
                INTERSTITIAL.getLiteral(), VIDEO_INTERSTITIAL_DSPS,
                BANNER.getLiteral(), VIDEO_BANNER_DSPS,
                REWARDED.getLiteral(), VIDEO_BANNER_DSPS
        );
    }

    public Set<Long> getDspTypes() {
        return dspTypes;
    }

    public void addDspType(CoreConstants.DspTypes dspType) {
        this.dspTypes.add(dspType.typeId);
    }

    public Set<Long> getVideoDspsForValidation(BaseBlock block) {
        if (block instanceof BlockWithBlockType) {
            return getVideoDsps(block);
        }

        return videoDspsForValidationByModel.getOrDefault(block.getClass(), Set.of());
    }

    public Map<Long, Dsp> getDspMap() {
        return dspMap;
    }

    public DspRuleContainer setDspMap(Map<Long, Dsp> dspMap) {
        this.dspMap = dspMap;
        return this;
    }

    public Map<Long, List<Dsp>> getBlockAvailableDspsMap() {
        return blockAvailableDspsMap;
    }

    public DspRuleContainer setBlockAvailableDspsMap(Map<Long, List<Dsp>> blockAvailableDspsMap) {
        this.blockAvailableDspsMap = blockAvailableDspsMap;
        return this;
    }

    public Set<Long> getVideoDsps(BaseBlock block) {
        if (block instanceof BlockWithBlockType) {
            return mobileVideoDspsMapByBlockType.getOrDefault(((BlockWithBlockType) block).getBlockType(), Set.of());
        }

        return videoDspsByModel.getOrDefault(block.getClass(), Set.of());
    }

    public Boolean canEditRichMedia() {
        return canEditRichMedia;
    }

    public void setCanEditRichMedia(Boolean canEditRichMedia) {
        this.canEditRichMedia = canEditRichMedia;
    }
}
