package ru.yandex.partner.core.entity.dsp.service;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import one.util.streamex.EntryStream;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.repository.filter.ConditionFilter;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.Query;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.container.BlockContainer;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithPageWithOwner;
import ru.yandex.partner.core.entity.dsp.filter.DspFilters;
import ru.yandex.partner.core.entity.dsp.model.BaseDsp;
import ru.yandex.partner.core.entity.dsp.model.Dsp;
import ru.yandex.partner.core.entity.dsp.repository.DspTypedRepository;
import ru.yandex.partner.core.entity.dsp.rules.DspRule;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleComponent;
import ru.yandex.partner.core.entity.dsp.rules.DspRuleContainer;
import ru.yandex.partner.core.entity.dsp.rules.result.DspRuleResult;
import ru.yandex.partner.core.entity.dsp.rules.result.DspRuleResultType;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.entity.tasks.doaction.DoActionFilterEnum;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.FilterOperator;

import static java.util.function.Predicate.not;
import static org.jooq.impl.DSL.groupConcat;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterModelsOfClass;
import static ru.yandex.partner.core.entity.dsp.rules.DspRuleUtils.buildAvailableDspsFilter;
import static ru.yandex.partner.core.entity.dsp.rules.DspRuleUtils.buildDefaultDspsFilter;
import static ru.yandex.partner.core.multistate.dsp.DspStateFlag.DELETED;
import static ru.yandex.partner.dbschema.partner.Tables.DSP;
import static ru.yandex.partner.dbschema.partner.Tables.DSP_FORMAT;
import static ru.yandex.partner.dbschema.partner.Tables.DSP_TYPE;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Service
@ParametersAreNonnullByDefault
public class DspService implements ModelQueryService<BaseDsp> {
    private final DspTypedRepository dspTypedRepository;
    private final DSLContext context;
    private final List<DspRuleComponent<? extends BaseBlock>> dspRuleComponentList;

    @Autowired
    public DspService(DSLContext context,
                      DspTypedRepository dspTypedRepository,
                      List<DspRuleComponent<? extends BaseBlock>> dspRuleComponentList) {
        this.dspTypedRepository = dspTypedRepository;
        this.context = context;
        this.dspRuleComponentList = dspRuleComponentList;
    }

    @Override
    public <S extends BaseDsp> List<S> findAll(Query<S> opts) {
        return filterModelsOfClass(
                dspTypedRepository.getAll(opts),
                opts.getClazz()
        );
    }

    @Override
    public <S extends BaseDsp> long count(Query<S> opts) {
        return dspTypedRepository.getEntityCountByCondition(opts.getFilter(), opts.getClazz());
    }

    @Override
    public ModelProperty<? extends Model, Long> getIdModelProperty() {
        return BaseDsp.ID;
    }

    @Override
    public Class<BaseDsp> getBaseClass() {
        return BaseDsp.class;
    }

    @Override
    public <S extends BaseDsp> MetaFilter<? super S, ?> getMetaFilterForDoAction(
            DoActionFilterEnum doActionFilterEnum
    ) {
        throw new UnsupportedOperationException("Dsp does not have required filters");
    }

    public List<Dsp> getAllNotDeletedDsps() {
        CoreFilterNode<Dsp> filterNode =
                CoreFilterNode.create(DspFilters.MULTISTATE,
                        FilterOperator.IN, not(has(DELETED)));
        List<BaseDsp> dsps = dspTypedRepository.getAll(QueryOpts.forClass(Dsp.class)
                .withFilter(filterNode)
        );

        return dsps.stream()
                .map(Dsp.class::cast)
                .collect(Collectors.toList());
    }

    public <B extends BlockWithPageWithOwner> List<List<Dsp>> getAvailableDsps(List<B> blocks,
                                                                               Predicate<B> canEditRichMedia) {
        List<DspRuleResult> dspRuleResults = new ArrayList<>(blocks.size());
        for (B block : blocks) {
            DspRuleComponent<? extends BaseBlock> dspRuleComponent = selectDspRuleComponentByClass(block.getClass());

            var dspRuleContainer = new DspRuleContainer();
            dspRuleContainer.setCanEditRichMedia(canEditRichMedia.test(block));
            dspRuleComponent.fillDspRuleContainer(dspRuleContainer, block);

            var dspRuleSet = dspRuleComponent.getDspRuleSet(dspRuleContainer);

            var dspRuleResult = buildAvailableDspsFilter(dspRuleContainer, block, dspRuleSet);

            dspRuleResults.add(dspRuleResult);
        }

        return getDspsByRuleResults(dspRuleResults);
    }

    public <B extends BlockWithPageWithOwner> List<List<Dsp>> getDefaultDsps(List<B> blocks) {
        List<DspRuleResult> dspRuleResults = new ArrayList<>(blocks.size());
        for (B block : blocks) {
            DspRuleComponent<? extends BaseBlock> dspRuleComponent = selectDspRuleComponentByClass(block.getClass());

            var dspRuleContainer = new DspRuleContainer();
            dspRuleComponent.fillDspRuleContainer(dspRuleContainer, block);

            var dspRuleSet = dspRuleComponent.getDspRuleSet(dspRuleContainer);

            BasePage page = block.getCampaign();

            var availableDspRuleResult = buildAvailableDspsFilter(dspRuleContainer, block, dspRuleSet);
            var defaultDspRuleResult = buildDefaultDspsFilter(dspRuleContainer, page, block, dspRuleSet);

            var dspRuleResult = defaultDspRuleResult.and(availableDspRuleResult);

            dspRuleResults.add(dspRuleResult);
        }

        return getDspsByRuleResults(dspRuleResults);
    }

    public ValidationResult<? extends BaseBlock, Defect> dspsFieldValidation(
            BlockContainer container,
            ValidationResult<? extends BaseBlock, Defect> vr) {
        BaseBlock value = vr.getValue();
        DspRuleComponent<? extends BaseBlock> dspRuleComponent = selectDspRuleComponentByClass(value.getClass());

        var dspRuleContainer = new DspRuleContainer();
        dspRuleComponent.fillDspRuleContainer(dspRuleContainer, value);

        var dspRuleSet = dspRuleComponent.getDspRuleSet(dspRuleContainer);
        for (var rule : dspRuleSet.getRules()) {
            rule.dspsFieldValidation(dspRuleContainer, container, vr);
        }
        return vr;
    }

    public <B extends BlockWithPageWithOwner> void dspsFieldEdit(
            List<? extends AppliedChanges<B>> appliedChanges,
            BlockContainer container
    ) {
        var dspIndex = getAllNotDeletedDsps().stream()
                .collect(Collectors.toMap(
                        Dsp::getId,
                        Function.identity()
                ));

        // TODO maybe get/write from/to block container?
        var blockAvailableDsps =
                getBlockAvailableDsps(Lists.transform(appliedChanges, AppliedChanges::getModel), container);

        for (AppliedChanges<? extends BaseBlock> appliedChange : appliedChanges) {
            BaseBlock block = appliedChange.getModel();
            DspRuleComponent<? extends BaseBlock> dspRuleComponent =
                    selectDspRuleComponentByClass(block.getClass());

            var dspRuleContainer = new DspRuleContainer();
            dspRuleContainer.setDspMap(dspIndex);
            dspRuleContainer.setBlockAvailableDspsMap(blockAvailableDsps);
            dspRuleComponent.fillDspRuleContainer(dspRuleContainer, block);

            var dspRuleSet = dspRuleComponent.getDspRuleSet(dspRuleContainer);

            for (DspRule rule : dspRuleSet.getRules()) {
                rule.dspsFieldEdit(dspRuleContainer, appliedChange);
            }
        }
    }

    public <B extends BlockWithPageWithOwner> Map<Long, List<Dsp>> getBlockAvailableDsps(List<B> blocks,
                                                                                         BlockContainer container) {
        Predicate<B> canEditRichMedia = (b) -> container.canEditRichMedia(b.getClass());
        List<List<Dsp>> availableDsps = getAvailableDsps(blocks, canEditRichMedia);

        return EntryStream.of(availableDsps)
                .keys()
                .toMap(k -> blocks.get(k).getId(), availableDsps::get);
    }

    public List<List<Dsp>> getDspsByRuleResults(List<DspRuleResult> dspRuleResults) {
        Map<String, List<BaseDsp>> customFilterQueryCache = new HashMap<>();
        var availableDsps = new ArrayList<List<Dsp>>();

        for (DspRuleResult dspRuleResult : dspRuleResults) {
            availableDsps.add(getDspsByRuleResult(dspRuleResult, customFilterQueryCache));
        }

        return availableDsps;
    }

    public List<Dsp> getDspsByRuleResult(DspRuleResult dspRuleResult,
                                         Map<String, List<BaseDsp>> customFilterQueryCache) {
        if (dspRuleResult.getType().equals(DspRuleResultType.LIMIT_ZERO)) {
            return List.of();
        } else {
            List<Field<?>> fields = new ArrayList<>(Arrays.asList(DSP.fields()));
            fields.add(groupConcat(DSP_TYPE.TYPE_ID));
            SelectQuery<Record> query =
                    context.select(fields).from(DSP).leftJoin(DSP_TYPE).on(DSP.ID.eq(DSP_TYPE.DSP_ID))
                            .leftJoin(DSP_FORMAT).on(DSP.ID.eq(DSP_FORMAT.DSP_ID))
                            .groupBy(DSP.ID)
                            .getQuery();
            query.addOrderBy(DSP.SHORT_CAPTION);
            List<BaseDsp> typed;
            if (dspRuleResult.getType().equals(DspRuleResultType.LIMIT_IDENTITY)) {
                typed = customFilterQueryCache.computeIfAbsent(
                        "",
                        k -> dspTypedRepository.getTyped(
                                context,
                                query,
                                null,
                                null,
                                BaseDsp.class,
                                false
                        ));
            } else {
                typed = customFilterQueryCache.computeIfAbsent(
                        // FIXME BEWARE LOCAL CACHE BY STRING QUERY REPRESENTATION
                        // i.e. jooq renders
                        dspRuleResult.getInnerCondition().toString(),
                        queryAsString -> dspTypedRepository.getTyped(
                                context,
                                query,
                                new ConditionFilter() {
                                    @Override
                                    protected Condition getCondition() {
                                        return dspRuleResult.getInnerCondition();
                                    }

                                    @Override
                                    public boolean isEmpty() {
                                        return false;
                                    }
                                },
                                null,
                                BaseDsp.class,
                                false
                        )
                );
            }
            return typed.stream().map(Dsp.class::cast).collect(Collectors.toList());
        }
    }

    public DspRuleComponent<? extends BaseBlock> selectDspRuleComponentByClass(Class<?> clazz) {
        DspRuleComponent<? extends BaseBlock> dspRuleComponent = dspRuleComponentList.stream()
                .filter(c -> c.getEntireTypeClass() != null)
                .filter(c -> c.getEntireTypeClass().equals(clazz))
                .findAny()
                .orElse(null);
        if (dspRuleComponent == null) {
            throw new RuntimeException("Found block with unregistered type in DSP rule component list");
        }
        return dspRuleComponent;
    }

    public <B extends BlockWithPageWithOwner> Map<String, Object> getDefaults(
            B block,
            boolean userCanViewDsps,
            Boolean pageIsUnmoderatedAuction,
            boolean canEditRichMedia
    ) {
        Map<String, Object> result = Maps.newHashMapWithExpectedSize(2);
        List<Dsp> dspsAvailable = getAvailableDsps(List.of(block), x -> canEditRichMedia).get(0);
        List<Dsp> dspsDefault = getDefaultDsps(List.of(block)).get(0);

        if (!userCanViewDsps) {
            if (Boolean.TRUE.equals(pageIsUnmoderatedAuction)) {
                dspsAvailable = dspsAvailable.stream()
                        .filter(dsp -> Boolean.TRUE.equals(dsp.getUnmoderatedRtbAuction()))
                        .collect(Collectors.toList());
                dspsDefault = dspsDefault.stream()
                        .filter(dsp -> Boolean.TRUE.equals(dsp.getUnmoderatedRtbAuction()))
                        .collect(Collectors.toList());
            } else {
                dspsAvailable = Collections.emptyList();
                dspsDefault = Collections.emptyList();
            }
        }

        result.put(
                "dsps_available",
                dspsAvailable.stream()
                        .map(this::dspToDto)
                        .collect(Collectors.toList())
        );
        result.put(
                "dsps_default",
                dspsDefault.stream()
                        .map(this::dspToDto).
                        collect(Collectors.toList())
        );

        return result;
    }

    private Map<String, Object> dspToDto(Dsp dsp) {
        Map<String, Object> dto = Maps.newHashMapWithExpectedSize(5);
        dto.put("id", Long.toString(dsp.getId()));
        dto.put("short_caption", dsp.getShortCaption());
        dto.put("types",
                Objects.requireNonNullElse(dsp.getTypes(), List.of())
                        .stream()
                        .map(Object::toString)
                        .collect(Collectors.toList())
        );
        dto.put("formats",
                Objects.requireNonNullElse(dsp.getFormats(), List.of())
                        .stream()
                        .map(Object::toString)
                        .collect(Collectors.toList()));
        if (dsp.getUnmoderatedRtbAuction()) {
            dto.put("unmoderated_rtb_auction", "1");
        }

        return dto;
    }
}
