package ru.yandex.partner.core.entity.dsp.type.owner;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.dsp.model.DspWithOwner;
import ru.yandex.partner.core.entity.dsp.repository.type.AbstractDspRepositoryTypeSupport;
import ru.yandex.partner.core.entity.dsp.repository.type.DspRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.entity.user.model.CommonUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.partner.core.entity.dsp.DspConstants.EDIT_FORBIDDEN_MODEL_PROPERTIES;
import static ru.yandex.partner.core.entity.dsp.model.prop.DspWithOwnerOwnerIdPropHolder.OWNER_ID;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.dbschema.partner.Tables.DSP;

@Component
@ParametersAreNonnullByDefault
public class DspWithOwnerRepositoryTypeSupport
        extends AbstractDspRepositoryTypeSupport<DspWithOwner>
        implements DspRepositoryTypeSupportWithMapper<DspWithOwner> {
    private final JooqMapper<DspWithOwner> mapper;
    private final Set<ModelProperty<? super DspWithOwner, String>> userProps;
    private final UserService userService;

    @Autowired
    protected DspWithOwnerRepositoryTypeSupport(DSLContext dslContext, UserService userService) {
        super(dslContext);
        this.userService = userService;
        this.mapper = createCommonDspMapper();
        this.userProps = Set.of(DspWithOwner.LOGIN);
    }

    private static JooqMapper<DspWithOwner> createCommonDspMapper() {
        return JooqMapperBuilder.<DspWithOwner>builder()
                .map(property(OWNER_ID, DSP.OWNER_ID))
                .build();
    }

    @Override
    public JooqMapper<DspWithOwner> getJooqMapper() {
        return mapper;
    }

    @Override
    public Class<DspWithOwner> getTypeClass() {
        return DspWithOwner.class;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(DspWithOwner model) {
        return fromModelProperties(
                new HashSet<>(
                        Sets.difference(getJooqMapper().getWritableModelProperties(), EDIT_FORBIDDEN_MODEL_PROPERTIES)
                )
        );
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<DspWithOwner> models) {
        var ownerIds = models.stream()
                .map(DspWithOwner::getOwnerId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        if (ownerIds.isEmpty()) {
            return;
        }

        var userLogins = userService.findAll(QueryOpts.forClass(User.class)
                .withFilterByIds(ownerIds)
                .withProps(Set.of(CommonUser.ID, CommonUser.LOGIN))
        ).stream().collect(Collectors.toMap(
                User::getId,
                User::getLogin
        ));

        models.forEach(model -> {
            if (model.getOwnerId() != null) {
                model.setLogin(userLogins.get(model.getOwnerId()));
            }
        });
    }

    @Override
    public Set<ModelProperty<? super DspWithOwner, ?>> getAffectedModelProperties() {
        return Sets.union(
                DspRepositoryTypeSupportWithMapper.super.getAffectedModelProperties(),
                userProps
        );
    }
}
