package ru.yandex.partner.core.entity.dsp.type.types;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.dsp.model.BaseDsp;
import ru.yandex.partner.core.entity.dsp.model.DspWithTypes;
import ru.yandex.partner.core.entity.dsp.repository.type.AbstractDspRepositoryTypeSupport;
import ru.yandex.partner.core.entity.dsp.repository.type.DspRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;
import ru.yandex.partner.libs.i18n.GettextMsg;

import static ru.yandex.partner.core.entity.dsp.DspConstants.DSP_TYPE_CAPTIONS;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.dbschema.partner.Tables.DSP_TYPE;

@Component
@ParametersAreNonnullByDefault
public class DspWithTypesRepositoryTypeSupport extends AbstractDspRepositoryTypeSupport<DspWithTypes>
        implements DspRepositoryTypeSupportWithoutMapper<DspWithTypes> {
    private final Set<ModelProperty<? super DspWithTypes, ?>> affectedModelProperties;
    private final Set<ModelProperty<? super DspWithTypes, ?>> editableModelProperties;

    @Autowired
    public DspWithTypesRepositoryTypeSupport(DSLContext dslContext) {
        super(dslContext);
        this.affectedModelProperties = Set.of(DspWithTypes.TYPES, DspWithTypes.TYPE_CAPTIONS);
        this.editableModelProperties = Set.of(DspWithTypes.TYPES);
    }

    @Override
    public Class<DspWithTypes> getTypeClass() {
        return DspWithTypes.class;
    }

    @Override
    public Set<ModelProperty<? super DspWithTypes, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(DspWithTypes model) {
        return fromModelProperties(new HashSet<>(editableModelProperties));
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<DspWithTypes> models) {
        Set<Long> modelIds = models.stream()
                .map(BaseDsp::getId)
                .collect(Collectors.toSet());
        Map<Long, List<Long>> selectedDspTypes = dslContext.select(DSP_TYPE.DSP_ID, DSP_TYPE.TYPE_ID)
                .from(DSP_TYPE)
                .where(DSP_TYPE.DSP_ID.in(modelIds))
                .fetchGroups(DSP_TYPE.DSP_ID, DSP_TYPE.TYPE_ID);
        for (DspWithTypes dsp : models) {
            List<Long> dspTypes = selectedDspTypes.getOrDefault(dsp.getId(), List.of());
            List<GettextMsg> dspCaptions = dspTypes.stream()
                    .map(DSP_TYPE_CAPTIONS::get)
                    .collect(Collectors.toList());
            dsp.setTypes(dspTypes);
            dsp.setTypeCaptions(dspCaptions);
        }
    }
}
