package ru.yandex.partner.core.entity.media;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.DSLContext;
import org.springframework.stereotype.Repository;

import ru.yandex.partner.core.entity.PageBlockIds;
import ru.yandex.partner.core.entity.utils.ConditionUtils;
import ru.yandex.partner.dbschema.partner.tables.records.MediaSizesRecord;
import ru.yandex.partner.libs.utils.JooqUtils;

import static ru.yandex.partner.dbschema.partner.tables.MediaSizes.MEDIA_SIZES;

@Repository
public class MediaSizeRepository {

    public Map<PageBlockIds, List<MediaSize>> getMediaSizes(DSLContext dslContext,
                                                            Collection<PageBlockIds> pageBlockIdsList) {
        if (pageBlockIdsList.isEmpty()) {
            return Map.of();
        }

        return dslContext.select(MEDIA_SIZES.PAGE_ID, MEDIA_SIZES.BLOCK_ID, MEDIA_SIZES.TYPE)
                .from(MEDIA_SIZES)
                .where(ConditionUtils.toPageBlockCondition(pageBlockIdsList, MEDIA_SIZES.PAGE_ID, MEDIA_SIZES.BLOCK_ID))
                .fetchGroups(
                        record -> new PageBlockIds(record.get(MEDIA_SIZES.PAGE_ID), record.get(MEDIA_SIZES.BLOCK_ID)),
                        record -> new MediaSize(
                                record.get(MEDIA_SIZES.PAGE_ID),
                                record.get(MEDIA_SIZES.BLOCK_ID),
                                record.get(MEDIA_SIZES.TYPE)
                        )
                );
    }

    public void updateMediaSizes(DSLContext dslContext,
                                 Map<PageBlockIds, List<String>> mediaSizesList) {
        dslContext.delete(MEDIA_SIZES)
                .where(ConditionUtils.toPageBlockCondition(mediaSizesList.keySet().stream()
                                .map(e -> new PageBlockIds(e.getPageId(), e.getBlockId()))
                                .collect(Collectors.toCollection(ArrayList::new)),
                        MEDIA_SIZES.PAGE_ID, MEDIA_SIZES.BLOCK_ID))
                .execute();

        addMediaSizes(dslContext, mediaSizesList);
    }

    public void addMediaSizes(DSLContext dslContext, Map<PageBlockIds, List<String>> mediaSizesList) {
        List<MediaSizesRecord> records = new ArrayList<>();
        for (Map.Entry<PageBlockIds, List<String>> entry : mediaSizesList.entrySet()) {
            for (String mediaSize : entry.getValue()) {
                records.add(new MediaSizesRecord(entry.getKey().getPageId(), entry.getKey().getBlockId(), mediaSize));
            }
        }
        JooqUtils.insertRecords(dslContext, MEDIA_SIZES, records);
    }

}
