package ru.yandex.partner.core.entity.notification;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.partner.core.entity.user.model.User;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.partner.dbschema.partner.tables.UserNotifications.USER_NOTIFICATIONS;

@Repository
public class UserNotificationsRepository {
    private final DSLContext dslContext;
    private final NotificationRepository notificationRepository;
    private final ObjectMapper objectMapper = new ObjectMapper();

    @Autowired
    public UserNotificationsRepository(
            DSLContext dslContext,
            NotificationRepository notificationRepository
    ) {
        this.dslContext = checkNotNull(dslContext);
        this.notificationRepository = notificationRepository;
    }

    public Map<Long, Long> getNotificationsCountByUserIds(Collection<Long> userIds) {
        if (userIds == null || userIds.isEmpty()) {
            return Collections.emptyMap();
        }

        return dslContext.select(USER_NOTIFICATIONS.USER_ID, DSL.count())
                .from(USER_NOTIFICATIONS)
                .where(USER_NOTIFICATIONS.USER_ID.in(userIds))
                .and(USER_NOTIFICATIONS.MULTISTATE.in(setViewedMultiStates()))
                .groupBy(USER_NOTIFICATIONS.USER_ID)
                .fetch()
                .intoMap(
                        record -> record.get(USER_NOTIFICATIONS.USER_ID),
                        record -> record.get(DSL.count()).longValue()
                );
    }


    public void createCpmConvertNotification(User user) {
        Long notificationId = notificationRepository.getConvertCpmNotificationId();

        dslContext.insertInto(USER_NOTIFICATIONS)
                .set(USER_NOTIFICATIONS.NOTIFICATION_ID, notificationId)
                .set(USER_NOTIFICATIONS.USER_ID, user.getId())
                .set(USER_NOTIFICATIONS.OPTS, getOptsWithCustomData(user))
                .set(USER_NOTIFICATIONS.CREATE_DATE, LocalDateTime.now())
                .execute();
    }

    private String getOptsWithCustomData(User user) {
        try {
            return objectMapper.writeValueAsString(
                    new Opts(new CustomData(user.getCurrentCurrency(), user.getCurrencyRate()))
            );
        } catch (JsonProcessingException e) {
            throw new RuntimeException(
                    "Can't serialize user %s fields to opts.custom_data".formatted(user.getId()), e
            );
        }
    }

    private record Opts(@JsonProperty("custom_data") CustomData customData) { }
    private record CustomData(String currency, @JsonProperty("currency_rate") BigDecimal currencyRate) { }

    private List<Long> setViewedMultiStates() {
        // Пока хардкод, работа с мультстатусами будет прорабатываться позже
        return List.of(0L);
    }
}
