package ru.yandex.partner.core.entity.page.multistate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.ContextPage;
import ru.yandex.partner.core.messages.PageActionMsg;
import ru.yandex.partner.core.messages.PageStateMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.page.ContextPageMultistate;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;
import ru.yandex.partner.libs.multistate.graph.PageMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.BALANCE_REGISTERED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.BLOCKED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_APPROVE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_UPDATE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.PROTECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.READ_ONLY;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.REJECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.STOPPED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.TESTING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.UPDATING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasOneOf;

@Component
@ParametersAreNonnullByDefault
public class ContextPageMultistateGraph extends
        AbstractMultistateGraph<ContextPage, PageStateFlag>
        implements PageMultistateGraph<ContextPage> {

    @Override
    public Multistate<PageStateFlag> getMultistateFromModel(ContextPage model) {
        return model.getMultistate();
    }

    @Override
    protected Multistate<PageStateFlag> getMultistateForValue(Long multistateValue) {
        return new ContextPageMultistate(multistateValue);
    }

    @Override
    @SuppressWarnings({"MethodLength", "LineLength"})
    protected Map<ActionNameHolder, ActionEntry<ContextPage, PageStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<ContextPage, PageStateFlag>> actionEntryMap = new HashMap<>();

        actionEntryMap.put(PageActionsEnum.ADD,
                getActionEntryBuilder(PageActionMsg.ADD, Set.of())
                        .setPredicate(empty())
                        .setSetFlags(Map.of(
                                NEED_APPROVE, true,
                                CHECK_STATISTICS, true
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.REGISTER_IN_BALANCE,
                getActionEntryBuilder(PageActionMsg.REGISTER_IN_BALANCE).
                        setPredicate(hasNoneOf(BALANCE_REGISTERED, DELETED, NEED_APPROVE, REJECTED, BLOCKED))
                        .setSetFlags(BALANCE_REGISTERED, true)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.START,
                getActionEntryBuilder(PageActionMsg.START)
                        .setPredicate(
                                has(BALANCE_REGISTERED).and(
                                        hasNoneOf(WORKING, DELETED, BLOCKED, NEED_APPROVE, REJECTED, PROTECTED)
                                )
                        )
                        .setSetFlags(
                                Map.of(
                                        WORKING, true,
                                        STOPPED, false,
                                        TESTING, false
                                )
                        )
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.START_TESTING,
                getActionEntryBuilder(PageActionMsg.START_TESTING)
                        .setPredicate(
                                has(BALANCE_REGISTERED).and(
                                        hasNoneOf(TESTING, WORKING, BLOCKED, NEED_APPROVE, REJECTED, PROTECTED)
                                )
                        )
                        .setSetFlags(
                                Map.of(
                                        TESTING, true,
                                        STOPPED, false,
                                        WORKING, false
                                )
                        )
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.STOP,
                getActionEntryBuilder(PageActionMsg.STOP)
                        .setPredicate(hasOneOf(WORKING, TESTING)
                                .and(hasNoneOf(STOPPED, NEED_APPROVE, REJECTED, PROTECTED)))
                        .setSetFlags(
                                Map.of(
                                        STOPPED, true,
                                        TESTING, false,
                                        WORKING, false
                                )
                        )
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.DELETE,
                getActionEntryBuilder(PageActionMsg.ARCHIVE)
                        .setPredicate(hasNoneOf(WORKING, TESTING, DELETED))
                        .setSetFlags(DELETED, true)
                        .build()

        );

        actionEntryMap.put(PageActionsEnum.RESTORE,
                getActionEntryBuilder(PageActionMsg.RESTORE)
                        .setPredicate(has(DELETED).and(not(has(BLOCKED))))
                        .setSetFlags(DELETED, false)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.EDIT,
                getActionEntryBuilder(PageActionMsg.EDIT)
                        .setPredicate(hasNoneOf(DELETED, BLOCKED, REJECTED))
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.DEL_MIRROR,
                getActionEntryBuilder(PageActionMsg.REMOVE_MIRROR)
                        .setPredicate(hasNoneOf(DELETED, BLOCKED, REJECTED, PROTECTED))
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.SET_READ_ONLY,
                getActionEntryBuilder(PageActionMsg.SET_READ_ONLY)
                        .setPredicate(not(has(READ_ONLY)))
                        .setSetFlags(READ_ONLY, true)
                        .build()
        );


        actionEntryMap.put(PageActionsEnum.RESET_READ_ONLY,
                getActionEntryBuilder(PageActionMsg.RESET_READ_ONLY)
                        .setPredicate(has(READ_ONLY))
                        .setSetFlags(READ_ONLY, false)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.SET_BLOCKED,
                getActionEntryBuilder(PageActionMsg.SET_BLOCKED)
                        .setPredicate(not(has(BLOCKED)))
                        .setSetFlags(Map.of(
                                BLOCKED, true,
                                NEED_APPROVE, false
                        ))
                        .build()
        );
        actionEntryMap.put(PageActionsEnum.RESET_BLOCKED,
                getActionEntryBuilder(PageActionMsg.RESET_BLOCKED)
                        .setPredicate(has(BLOCKED))
                        .setSetFlags(BLOCKED, false)
                        .build()
        );
        actionEntryMap.put(PageActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(CHECK_STATISTICS))
                        .setSetFlags(CHECK_STATISTICS, false)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.RESTORE_BLOCK,
                getActionEntryBuilder(PageActionMsg.RESTORE_BLOCK_ON_CAMPAIGN)
                        .setPredicate(not(has(DELETED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.START_BLOCK,
                getActionEntryBuilder(PageActionMsg.START_BLOCK_ON_CAMPAIGN)
                        .setPredicate(hasOneOf(WORKING, TESTING).and(not(has(PROTECTED))))
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(PageStateMsg.NEED_UPDATE)
                        .setPredicate(hasNoneOf(NEED_APPROVE, REJECTED, DELETED))
                        .setSetFlags(NEED_UPDATE, true)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.APPROVE,
                getActionEntryBuilder(PageActionMsg.APPROVE)
                        .setPredicate(has(NEED_APPROVE))
                        .setSetFlags(NEED_APPROVE, false)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.REJECT,
                getActionEntryBuilder(PageActionMsg.REJECT)
                        // TODO: добавить проверку, что пейдж не protected
                        //  https://github.yandex-team.ru/partner/partner2/pull/5575/files#diff-d7ff66101c83a49fc7fd2f382d29827e0450d3f0a107abe85b1520a2ec21fd21R410
                        .setPredicate(not(has(REJECTED)))
                        .setSetFlags(Map.of(
                                REJECTED, true,
                                NEED_APPROVE, false
                        ))
                        .build()
        );
        actionEntryMap.put(PageActionsEnum.CAN_UPDATE_IN_BK,
                getActionEntryBuilder(PageActionMsg.CAN_UPDATE_IN_BK)
                        .setPredicate(hasNoneOf(NEED_APPROVE, REJECTED, DELETED, PROTECTED))
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.SET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.SET_PROTECTED)
                        .setPredicate(not(has(PROTECTED)))
                        .setSetFlags(PROTECTED, true)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.RESET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.RESET_PROTECTED)
                        .setPredicate(has(PROTECTED))
                        .setSetFlags(PROTECTED, false)
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.START_UPDATE,
                getActionEntryBuilder(PageActionMsg.START_UPDATE)
                        .setPredicate(hasOneOf(NEED_UPDATE, UPDATING))
                        .setSetFlags(Map.of(
                                UPDATING, true,
                                NEED_UPDATE, false
                        ))
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(PageActionMsg.STOP_UPDATE)
                        .setPredicate(has(UPDATING))
                        .setSetFlags(UPDATING, false)
                        .build()
        );
        return actionEntryMap;
    }

    private ActionEntry.Builder<ContextPage, PageStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(ContextPage.ID, ContextPage.MULTISTATE));
    }

    @Override
    public Class<ContextPage> getModelClass() {
        return ContextPage.class;
    }

    @Override
    public ContextPageMultistate convertMultistate(List<PageStateFlag> enabledFlags) {
        return new ContextPageMultistate(enabledFlags);
    }
}
