package ru.yandex.partner.core.entity.page.multistate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.InternalContextPage;
import ru.yandex.partner.core.messages.PageActionMsg;
import ru.yandex.partner.core.messages.PageStateMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.page.InternalContextPageMultistate;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;
import ru.yandex.partner.libs.multistate.graph.PageMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.BALANCE_REGISTERED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_UPDATE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.PROTECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.READ_ONLY;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.STOPPED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.TESTING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.UPDATING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasOneOf;


@Component
@ParametersAreNonnullByDefault
public class InternalContextPageMultistateGraph
        extends AbstractMultistateGraph<InternalContextPage, PageStateFlag>
        implements PageMultistateGraph<InternalContextPage> {

    @Override
    protected Multistate<PageStateFlag> getMultistateForValue(Long multistateValue) {
        return new InternalContextPageMultistate(multistateValue);
    }

    @Override
    public Multistate<PageStateFlag> getMultistateFromModel(InternalContextPage model) {
        return model.getMultistate();
    }

    @Override
    public Class<InternalContextPage> getModelClass() {
        return InternalContextPage.class;
    }

    @Override
    public InternalContextPageMultistate convertMultistate(List<PageStateFlag> enabledFlags) {
        return new InternalContextPageMultistate(enabledFlags);
    }

    @Override
    @SuppressWarnings("MethodLength")
    protected Map<ActionNameHolder, ActionEntry<InternalContextPage, PageStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<InternalContextPage, PageStateFlag>> actionEntryMap = new HashMap<>();

        actionEntryMap.put(PageActionsEnum.ADD,
                getActionEntryBuilder(PageActionMsg.ADD, Set.of())
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(PageActionsEnum.REGISTER_IN_BALANCE,
                getActionEntryBuilder(PageActionMsg.REGISTER_IN_BALANCE)
                        .setPredicate(hasNoneOf(BALANCE_REGISTERED, DELETED))
                        .setSetFlags(BALANCE_REGISTERED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.START,
                getActionEntryBuilder(PageActionMsg.START)
                        .setPredicate(has(BALANCE_REGISTERED).and(hasNoneOf(WORKING, DELETED, PROTECTED)))
                        .setSetFlags(Map.of(WORKING, true, STOPPED, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.START_TESTING,
                getActionEntryBuilder(PageActionMsg.START_TESTING)
                        .setPredicate(has(STOPPED).and(not(has(PROTECTED))))
                        .setSetFlags(Map.of(TESTING, true, STOPPED, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP_TESTING,
                getActionEntryBuilder(PageActionMsg.STOP_TESTING)
                        .setPredicate(has(TESTING).and(not(has(PROTECTED))))
                        .setSetFlags(Map.of(STOPPED, true, TESTING, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP,
                getActionEntryBuilder(PageActionMsg.STOP)
                        .setPredicate(hasOneOf(WORKING, TESTING).and(hasNoneOf(STOPPED, PROTECTED)))
                        .setSetFlags(Map.of(STOPPED, true, WORKING, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.DELETE,
                getActionEntryBuilder(PageActionMsg.ARCHIVE)
                        .setPredicate(hasNoneOf(WORKING, TESTING, DELETED))
                        .setSetFlags(DELETED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE,
                getActionEntryBuilder(PageActionMsg.RESTORE)
                        .setPredicate(has(DELETED).and(not(has(PROTECTED))))
                        .setSetFlags(Map.of(
                                WORKING, true,
                                DELETED, false,
                                STOPPED, false,
                                TESTING, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.EDIT,
                getActionEntryBuilder(PageActionMsg.EDIT)
                        .setPredicate(not(has(DELETED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.DEL_MIRROR,
                getActionEntryBuilder(PageActionMsg.REMOVE_MIRROR)
                        .setPredicate(hasNoneOf(DELETED, PROTECTED))
                        .build());


        actionEntryMap.put(PageActionsEnum.SET_READ_ONLY,
                getActionEntryBuilder(PageActionMsg.SET_READ_ONLY)
                        .setPredicate(not(has(READ_ONLY)))
                        .setSetFlags(READ_ONLY, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_READ_ONLY,
                getActionEntryBuilder(PageActionMsg.RESET_READ_ONLY)
                        .setPredicate(has(READ_ONLY))
                        .setSetFlags(READ_ONLY, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.SET_PROTECTED)
                        .setPredicate(not(has(PROTECTED)))
                        .setSetFlags(PROTECTED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.RESET_PROTECTED)
                        .setPredicate(has(PROTECTED))
                        .setSetFlags(PROTECTED, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.SET_CHECK_STATISTICS)
                        .setPredicate(empty())
                        .setSetFlags(CHECK_STATISTICS, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(CHECK_STATISTICS))
                        .setSetFlags(CHECK_STATISTICS, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_BLOCK,
                getActionEntryBuilder(PageActionMsg.START_BLOCK_ON_CAMPAIGN)
                        .setPredicate(hasOneOf(WORKING, TESTING).and(not(has(PROTECTED))))
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE_BLOCK,
                getActionEntryBuilder(PageActionMsg.RESTORE_BLOCK_ON_CAMPAIGN)
                        .setPredicate(not(has(DELETED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.BK_IMPORT_WORKING,
                getActionEntryBuilder(PageActionMsg.BK_IMPORT_WORKING)
                        .setPredicate(not(has(WORKING)))
                        .setSetFlags(Map.of(
                                WORKING, true,
                                TESTING, false,
                                DELETED, false
                        ))
                        .build());
        actionEntryMap.put(PageActionsEnum.BK_IMPORT_TESTING,
                getActionEntryBuilder(PageActionMsg.BK_IMPORT_TESTING)
                        .setPredicate(not(has(TESTING)))
                        .setSetFlags(Map.of(
                                TESTING, true,
                                WORKING, false,
                                DELETED, false
                        ))
                        .build());
        actionEntryMap.put(PageActionsEnum.BK_IMPORT_DELETED,
                getActionEntryBuilder(PageActionMsg.BK_IMPORT_DELETED)
                        .setPredicate(not(has(DELETED)))
                        .setSetFlags(Map.of(
                                DELETED, true,
                                WORKING, false,
                                TESTING, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.CAN_UPDATE_IN_BK,
                getActionEntryBuilder(PageActionMsg.CAN_UPDATE_IN_BK)
                        .setPredicate(hasNoneOf(DELETED, PROTECTED))
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(PageStateMsg.NEED_UPDATE)
                        .setPredicate(hasNoneOf(DELETED))
                        .setSetFlags(NEED_UPDATE, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_UPDATE,
                getActionEntryBuilder(PageActionMsg.START_UPDATE)
                        .setPredicate(hasOneOf(NEED_UPDATE, UPDATING))
                        .setSetFlags(Map.of(
                                UPDATING, true,
                                NEED_UPDATE, false
                        ))
                        .build());
        actionEntryMap.put(PageActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(PageActionMsg.STOP_UPDATE)
                        .setPredicate(has(UPDATING))
                        .setSetFlags(UPDATING, false)
                        .build());

        return actionEntryMap;
    }

    private ActionEntry.Builder<InternalContextPage, PageStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(InternalContextPage.ID, InternalContextPage.MULTISTATE));
    }


}

