package ru.yandex.partner.core.entity.page.multistate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.InternalMobileApp;
import ru.yandex.partner.core.messages.PageActionMsg;
import ru.yandex.partner.core.messages.PageStateMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.page.InternalMobileAppMultistate;
import ru.yandex.partner.core.multistate.page.PageMultistate;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;
import ru.yandex.partner.libs.multistate.graph.PageMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_UPDATE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.PROTECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.STOPPED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.UPDATING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;

@Component
public class InternalMobileAppMultistateGraph extends AbstractMultistateGraph<InternalMobileApp, PageStateFlag>
        implements PageMultistateGraph<InternalMobileApp> {
    @Override
    protected Map<ActionNameHolder, ActionEntry<InternalMobileApp, PageStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<InternalMobileApp, PageStateFlag>> actionEntryMap = new HashMap<>();

        actionEntryMap.put(PageActionsEnum.ADD,
                getActionEntryBuilder(PageActionMsg.ADD, Set.of())
                        .setPredicate(empty())
                        .setSetFlags(Map.of(
                                STOPPED, true
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.START,
                getActionEntryBuilder(PageActionMsg.START)
                        .setPredicate(STOPPED.and(not(DELETED.or(PROTECTED))))
                        .setSetFlags(Map.of(
                                WORKING, true,
                                STOPPED, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP,
                getActionEntryBuilder(PageActionMsg.STOP)
                        .setPredicate(WORKING.and(not(STOPPED.or(PROTECTED))))
                        .setSetFlags(Map.of(
                                STOPPED, true,
                                WORKING, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.DELETE,
                getActionEntryBuilder(PageActionMsg.ARCHIVE)
                        .setPredicate(hasNoneOf(WORKING, DELETED))
                        .setSetFlags(DELETED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE,
                getActionEntryBuilder(PageActionMsg.RESTORE)
                        .setPredicate(DELETED.and(not(PROTECTED)))
                        .setSetFlags(Map.of(
                                WORKING, true,
                                DELETED, false,
                                STOPPED, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.EDIT,
                getActionEntryBuilder(PageActionMsg.EDIT)
                        .setPredicate(not(DELETED))
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.SET_CHECK_STATISTICS)
                        .setPredicate(empty())
                        .setSetFlags(CHECK_STATISTICS, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(CHECK_STATISTICS)
                        .setSetFlags(CHECK_STATISTICS, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_BLOCK,
                getActionEntryBuilder(PageActionMsg.START_BLOCK_ON_CAMPAIGN)
                        .setPredicate(WORKING.and(not(PROTECTED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE_BLOCK,
                getActionEntryBuilder(PageActionMsg.RESTORE_BLOCK_ON_CAMPAIGN)
                        .setPredicate(not(DELETED))
                        .build());

        actionEntryMap.put(PageActionsEnum.CAN_UPDATE_IN_BK,
                getActionEntryBuilder(PageActionMsg.CAN_UPDATE_IN_BK)
                        .setPredicate(not(DELETED.or(PROTECTED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.SET_PROTECTED)
                        .setPredicate(not(PROTECTED))
                        .setSetFlags(PROTECTED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.RESET_PROTECTED)
                        .setPredicate(PROTECTED)
                        .setSetFlags(PROTECTED, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(PageStateMsg.NEED_UPDATE)
                        .setPredicate(not(DELETED))
                        .setSetFlags(NEED_UPDATE, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_UPDATE,
                getActionEntryBuilder(PageActionMsg.START_UPDATE)
                        .setPredicate(NEED_UPDATE.or(UPDATING))
                        .setSetFlags(Map.of(
                                UPDATING, true,
                                NEED_UPDATE, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(PageActionMsg.STOP_UPDATE)
                        .setPredicate(has(UPDATING))
                        .setSetFlags(UPDATING, false)
                        .build());

        return actionEntryMap;
    }

    private ActionEntry.Builder<InternalMobileApp, PageStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(InternalMobileApp.ID, InternalMobileApp.MULTISTATE));
    }

    @Override
    protected Multistate<PageStateFlag> getMultistateForValue(Long multistateValue) {
        return new InternalMobileAppMultistate(multistateValue);
    }

    @Override
    public Multistate<PageStateFlag> getMultistateFromModel(InternalMobileApp model) {
        return InternalMobileApp.MULTISTATE.get(model);
    }

    @Override
    public Class<InternalMobileApp> getModelClass() {
        return InternalMobileApp.class;
    }

    @Override
    public PageMultistate convertMultistate(List<PageStateFlag> enabledFlags) {
        return new InternalMobileAppMultistate(enabledFlags);
    }
}
