package ru.yandex.partner.core.entity.page.multistate;


import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.page.actions.PageActionsEnum;
import ru.yandex.partner.core.entity.page.model.MobileAppSettings;
import ru.yandex.partner.core.messages.PageActionMsg;
import ru.yandex.partner.core.messages.PageStateMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.page.MobileAppSettingsMultistate;
import ru.yandex.partner.core.multistate.page.PageMultistate;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;
import ru.yandex.partner.libs.multistate.graph.PageMultistateGraph;


import static java.util.function.Predicate.not;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.BALANCE_REGISTERED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.BLOCKED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.CHECK_STATISTICS;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.DELETED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_APPROVE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.NEED_UPDATE;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.PROTECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.REJECTED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.STOPPED;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.TESTING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.UPDATING;
import static ru.yandex.partner.core.multistate.page.PageStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasOneOf;


@Component
@ParametersAreNonnullByDefault
public class MobileAppMultistateGraph extends AbstractMultistateGraph<MobileAppSettings, PageStateFlag>
        implements PageMultistateGraph<MobileAppSettings> {

    @Override
    protected Map<ActionNameHolder, ActionEntry<MobileAppSettings, PageStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<MobileAppSettings, PageStateFlag>> actionEntryMap = new HashMap<>();
        actionEntryMap.put(PageActionsEnum.ADD,
                getActionEntryBuilder(PageActionMsg.ADD, Set.of())
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_CHECK_STATISTICS,
                getActionEntryBuilder(PageActionMsg.RESET_CHECK_STATISTICS)
                        .setPredicate(has(CHECK_STATISTICS))
                        .setSetFlags(CHECK_STATISTICS, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_NEED_APPROVE,
        getActionEntryBuilder(PageActionMsg.SET_NEED_APPROVE)
                .setPredicate(
                        (has(REJECTED).and(not(has(BLOCKED))))
                                .or(hasNoneOf(WORKING, TESTING, STOPPED, DELETED, BLOCKED, PROTECTED, NEED_APPROVE))
                                .or(empty()))
                .setSetFlags(Map.of(NEED_APPROVE, true, REJECTED, false))
                .build());

        actionEntryMap.put(PageActionsEnum.REJECT,
                getActionEntryBuilder(PageActionMsg.REJECT)
                        .setPredicate(not(has(REJECTED)))
                        .setSetFlags(NEED_APPROVE, false)
                        .setSetFlags(REJECTED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.APPROVE,
                getActionEntryBuilder(PageActionMsg.APPROVE)
                        .setPredicate(has(NEED_APPROVE))
                        .setSetFlags(NEED_APPROVE, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_TESTING,
                getActionEntryBuilder(PageActionMsg.START_TESTING)
                        .setPredicate(has(BALANCE_REGISTERED).and(
                                hasNoneOf(TESTING, WORKING, DELETED, PROTECTED, NEED_APPROVE, REJECTED, BLOCKED)))
                        .setSetFlags(Map.of(TESTING, true, STOPPED, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP_TESTING,
                getActionEntryBuilder(PageActionMsg.STOP_TESTING)
                        .setPredicate(has(TESTING).and(not(has(PROTECTED))))
                        .setSetFlags(Map.of(STOPPED, true, TESTING, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.START,
                getActionEntryBuilder(PageActionMsg.START)
                        .setPredicate(has(BALANCE_REGISTERED).and(
                                hasNoneOf(WORKING, DELETED, PROTECTED, NEED_APPROVE, REJECTED, BLOCKED)))
                        .setSetFlags(Map.of(WORKING, true, TESTING, false, STOPPED, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP,
                getActionEntryBuilder(PageActionMsg.STOP)
                        .setPredicate(has(WORKING).and(not(has(PROTECTED))))
                        .setSetFlags(Map.of(STOPPED, true, WORKING, false, TESTING, false))
                        .build());

        actionEntryMap.put(PageActionsEnum.DELETE,
                getActionEntryBuilder(PageActionMsg.ARCHIVE)
                        .setPredicate(not(has(DELETED)).and(hasOneOf(NEED_APPROVE, STOPPED, REJECTED)))
                        .setSetFlags(DELETED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE,
                getActionEntryBuilder(PageActionMsg.RESTORE)
                        .setPredicate(has(DELETED).and(not(has(BLOCKED))))
                        .setSetFlags(DELETED, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.EDIT,
                getActionEntryBuilder(PageActionMsg.EDIT)
                        .setPredicate(hasNoneOf(DELETED, BLOCKED, REJECTED))
                        .build());

        actionEntryMap.put(PageActionsEnum.START_BLOCK,
                getActionEntryBuilder(PageActionMsg.START_BLOCK_ON_CAMPAIGN)
                        .setPredicate(hasOneOf(WORKING, TESTING).and(not(has(PROTECTED))))
                        .build());

        actionEntryMap.put(PageActionsEnum.RESTORE_BLOCK,
                getActionEntryBuilder(PageActionMsg.RESTORE_BLOCK_ON_CAMPAIGN)
                        .setPredicate(not(has(DELETED)))
                        .build());

        actionEntryMap.put(PageActionsEnum.CAN_UPDATE_IN_BK,
                getActionEntryBuilder(PageActionMsg.CAN_UPDATE_IN_BK)
                        .setPredicate(hasNoneOf(DELETED, PROTECTED))
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.SET_PROTECTED)
                        .setPredicate(not(has(PROTECTED)))
                        .setSetFlags(PROTECTED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.RESET_PROTECTED,
                getActionEntryBuilder(PageActionMsg.RESET_PROTECTED)
                        .setPredicate(has(PROTECTED))
                        .setSetFlags(PROTECTED, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_NEED_UPDATE,
                getActionEntryBuilder(PageStateMsg.NEED_UPDATE)
                        .setPredicate(hasNoneOf(DELETED))
                        .setSetFlags(NEED_UPDATE, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.START_UPDATE,
                getActionEntryBuilder(PageActionMsg.START_UPDATE)
                        .setPredicate(hasOneOf(NEED_UPDATE, UPDATING))
                        .setSetFlags(Map.of(
                                UPDATING, true,
                                NEED_UPDATE, false
                        ))
                        .build());

        actionEntryMap.put(PageActionsEnum.STOP_UPDATE,
                getActionEntryBuilder(PageActionMsg.STOP_UPDATE)
                        .setPredicate(has(UPDATING))
                        .setSetFlags(UPDATING, false)
                        .build());

        actionEntryMap.put(PageActionsEnum.REGISTER_IN_BALANCE,
                getActionEntryBuilder(PageActionMsg.REGISTER_IN_BALANCE)
                        .setPredicate(hasNoneOf(BALANCE_REGISTERED, DELETED, NEED_APPROVE, REJECTED, BLOCKED))
                        .setSetFlags(BALANCE_REGISTERED, true)
                        .build());

        actionEntryMap.put(PageActionsEnum.SET_BLOCKED,
                getActionEntryBuilder(PageActionMsg.SET_BLOCKED)
                        .setPredicate(not(has(BLOCKED)))
                        .setSetFlags(Map.of(
                                BLOCKED, true,
                                NEED_APPROVE, false
                        ))
                        .build()
        );

        actionEntryMap.put(PageActionsEnum.RESET_BLOCKED,
                getActionEntryBuilder(PageActionMsg.RESET_BLOCKED)
                        .setPredicate(has(BLOCKED))
                        .setSetFlags(BLOCKED, false)
                        .build()
        );
        return actionEntryMap;
    }

    private ActionEntry.Builder<MobileAppSettings, PageStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(MobileAppSettings.ID, MobileAppSettings.MULTISTATE));
    }


    @Override
    protected Multistate<PageStateFlag> getMultistateForValue(Long multistateValue) {
        return new MobileAppSettingsMultistate(multistateValue);
    }

    @Override
    public Multistate<PageStateFlag> getMultistateFromModel(MobileAppSettings model) {
        return model.getMultistate();
    }

    @Override
    public Class<MobileAppSettings> getModelClass() {
        return MobileAppSettings.class;
    }

    @Override
    public PageMultistate convertMultistate(List<PageStateFlag> enabledFlags) {
        return new MobileAppSettingsMultistate(enabledFlags);
    }
}
