package ru.yandex.partner.core.entity.page.repository.type;

import java.util.List;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.Nullable;
import org.jooq.Record;
import org.jooq.SelectQuery;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.partner.core.entity.page.container.PageOperationContainer;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.modelmeta.ModelMetaCollector;
import ru.yandex.partner.core.multitype.repository.PartnerRepositoryTypeSupportFacade;
import ru.yandex.partner.core.page.PageModelTypesHolder;
import ru.yandex.partner.core.page.PageType;

import static ru.yandex.partner.core.entity.page.service.PageUpdateOperationConstants.PAGE_TYPE_SUPPORT_CLASS_WHITE_LIST;
import static ru.yandex.partner.dbschema.partner.Tables.PAGES;

@Component
@ParametersAreNonnullByDefault
public class PageRepositoryTypeSupportFacade extends PartnerRepositoryTypeSupportFacade<
        BasePage, PageType, PageOperationContainer, PageOperationContainer> {

    private final PageModelTypesHolder typesHolder;

    @Autowired
    public PageRepositoryTypeSupportFacade(
            List<PageRepositoryTypeSupport<? extends BasePage, PageOperationContainer>> typeSupports,
            PageModelTypesHolder typesHolder) {
        super(
                typeSupports,
                ModelMetaCollector.getModelMetaHolders()
                        .stream()
                        .filter(modelMetaHolder -> BasePage.class.isAssignableFrom(modelMetaHolder.getEntityClass()))
                        .map(modelMetaHolder -> {
                            var name = modelMetaHolder.getName();
                            var pageType = PageType.from(name);
                            if (pageType == null) {
                                throw new RuntimeException("Cannot find PageType for " + name);
                            }
                            if (pageType.getConstructor() == null) {
                                throw new RuntimeException("Constructor cannot be null. Model = " + name);
                            }
                            return pageType;
                        })
                        .collect(Collectors.toMap(Function.identity(), PageType::getConstructor)),
                Set.of(PAGES.MODEL),
                PAGE_TYPE_SUPPORT_CLASS_WHITE_LIST);
        this.typesHolder = typesHolder;
    }

    @Override
    protected PageType getModelType(Record record) {
        PageType pageType = PageType.from(record.get(PAGES.MODEL).getLiteral());

        if (pageType == null) {
            throw new IllegalStateException("DB returned unknown BlockType. Record: " + record);
        }

        return pageType;
    }

    @Override
    protected void addModelConditionsToQuery(SelectQuery<?> query, @Nullable Class<? extends BasePage> clazz) {
        if (clazz == null) {
            return;
        }

        var type = typesHolder.getTypeForDb(clazz);
        if (type != null) {
            query.addConditions(PAGES.MODEL.eq(type));
        }
    }
}
