package ru.yandex.partner.core.entity.page.service;

import java.util.List;
import java.util.Set;

import org.jetbrains.annotations.NotNull;

import ru.yandex.direct.model.Model;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.partner.core.action.ActionContextFacade;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.page.filter.PageFilters;
import ru.yandex.partner.core.entity.page.model.BasePage;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.meta.MultistateMetaFilter;
import ru.yandex.partner.core.multistate.page.PageStateFlag;
import ru.yandex.partner.core.utils.ModelPropertyUtils;
import ru.yandex.partner.core.utils.OrderBy;

import static ru.yandex.partner.core.filter.operator.FilterOperator.IN;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.hasNoneOf;

public abstract class BaseReachablePageService<B extends BaseBlock> implements ReachablePageService<B> {
    private final ModelQueryService<BasePage> pageService;
    private final Set<ModelProperty<? extends Model, ?>> allModelProperties;
    private final ActionContextFacade actionContextFacade;

    public BaseReachablePageService(ModelQueryService<BasePage> pageService,
                                    Set<ModelProperty<?, ?>> allModelProperties,
                                    ActionContextFacade actionContextFacade) {
        this.pageService = pageService;
        this.allModelProperties = ModelPropertyUtils.castExtends(allModelProperties);
        this.actionContextFacade = actionContextFacade;
    }

    @Override
    public <T extends BasePage> List<T> getReachablePages(
            @NotNull CoreFilterNode<? super T> baseFilter,
            Class<T> clazz
    ) {
        return getReachablePages(
                baseFilter,
                null,
                allModelProperties,
                null,
                clazz
        );
    }

    @Override
    public <T extends BasePage> List<T> getReachablePagesForAdd(@NotNull CoreFilterNode<? super T> baseFilter,
                                                                Class<T> clazz) {
        return getReachablePagesForAdd(
                baseFilter,
                null,
                allModelProperties,
                null,
                clazz
        );
    }

    @Override
    public <T extends BasePage> List<T> getReachablePagesForAdd(@NotNull CoreFilterNode<? super T> baseFilter,
                                                                LimitOffset limitOffset,
                                                                Set<ModelProperty<? extends Model, ?>> modelProperties,
                                                                List<OrderBy> orderByList,
                                                                Class<T> clazz) {
        return getReachablePages(
                ((CoreFilterNode<T>) baseFilter).and(pageInActiveState()),
                limitOffset,
                allModelProperties,
                orderByList,
                clazz
        );
    }

    @Override
    public <T extends BasePage> List<T> getReachablePages(
            @NotNull CoreFilterNode<? super T> baseFilter,
            LimitOffset limitOffset,
            Set<ModelProperty<? extends Model, ?>> modelProperties,
            List<OrderBy> orderByList,
            Class<T> clazz) {

        var pageFilter = ((CoreFilterNode<T>) baseFilter)
                .and(customPageFilter());

        List<T> pages = pageService.findAll(
                QueryOpts.forClass(clazz)
                        .withFilter(pageFilter)
                        .withProps(modelProperties)
                        .withOrder(orderByList)
                        .withLimitOffset(limitOffset)
        );

        actionContextFacade.getActionContext(clazz).preloadModels(modelProperties, pages);
        return pages;
    }

    @NotNull
    protected abstract <T extends BasePage> CoreFilterNode<T> customPageFilter();

    protected MultistateMetaFilter<BasePage, PageStateFlag> getMultistateMetaFilter() {
        return PageFilters.MULTISTATE;
    }

    @NotNull
    private <T extends BasePage> CoreFilterNode<T> pageInActiveState() {
        var excludedStates = PageStateFlag.INACTIVE_STATES;
        if (excludedStates.isEmpty()) {
            // в любом статусе
            return CoreFilterNode.neutral();
        }

        return CoreFilterNode.create(
                getMultistateMetaFilter(), IN, hasNoneOf(excludedStates)
        );
    }

    @Override
    public <T extends BasePage> MetaFilter<? super T, Long> getPageIdMetaFilter() {
        return PageFilters.PAGE_ID;
    }
}
