package ru.yandex.partner.core.entity.page.type.bkdata

import NPartner.Page.TPartnerPage
import org.jooq.DSLContext
import org.jooq.Field
import org.jooq.Table
import org.jooq.impl.DSL
import org.springframework.stereotype.Component
import ru.yandex.direct.validation.builder.Constraint
import ru.yandex.direct.validation.builder.ItemValidationBuilder
import ru.yandex.direct.validation.builder.Validator
import ru.yandex.direct.validation.builder.When
import ru.yandex.direct.validation.constraint.CommonConstraints.inSet
import ru.yandex.direct.validation.result.Defect
import ru.yandex.partner.core.CoreConstants
import ru.yandex.partner.core.action.exception.DefectInfoWithMsgId.defect
import ru.yandex.partner.core.block.BlockType
import ru.yandex.partner.core.entity.block.service.OperationMode
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.DUPLICATE_SSP_TOKEN
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.EXCESS_SSP_PARAMS
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INCORRECT_LANG
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INCORRECT_SSP
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INCORRECT_STORE
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INVALID_CPA
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.INVALID_RELOAD_TIMEOUT
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.PAGE_VIDEO_NOT_SUPPORTED
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.SSP_REQUIRED
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.STRIPE_NOT_DEFINED
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.UNMODIFIABLE_PAGE
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.WRONG_PRODUCT_TYPE
import ru.yandex.partner.core.entity.block.service.validation.defects.presentation.BkDataValidationMsg.WRONG_VIDEO_TYPE
import ru.yandex.partner.core.entity.block.type.custombkdata.BlockWithCustomBkDataValidatorProvider
import ru.yandex.partner.core.entity.dsp.model.Dsp
import ru.yandex.partner.dbschema.partner.Partner.PARTNER
import java.util.regex.Pattern

@Component
class PageBkDataValidatorProvider(
    private val dslContext: DSLContext,
    private val blockBkDataValidatorProvider: BlockWithCustomBkDataValidatorProvider,
    private val slotValidatorProvider: SlotValidatorProvider,
) {
    private val notInternalPages = setOf(242L, 245L)
    private val knownProductTypes = setOf("video", "mobile", "search", "context", "indoor", "outdoor", "distribution")
    private val knownLangs = setOf("ru", "uk", "en", "by", "kz", "tr", "tt")
    private val knownStores = setOf("GooglePlay", "AppStore")

    fun validator(mode: OperationMode, allDsps: Map<Long, Dsp>) =
        Validator<TPartnerPage, Defect<Any>> { pageBkData ->
            val vb = ItemValidationBuilder.of<TPartnerPage, Defect<Any>>(pageBkData)

            val pageIsSsp = pageToSspLinkModels.contains(pageBkData.productID)
            if (pageIsSsp) {
                if (!pageBkData.hasSSPID() && pageBkData.sspPageTokenList.isEmpty()) {
                    vb.item(pageBkData.sspid, "SSPID")
                        .check { defect(SSP_REQUIRED) }

                    return@Validator vb.result
                }

                vb.list(pageBkData.sspPageTokenList, "SSPPageToken")
                    .checkEach(Constraint { token ->
                        if (token.isNullOrBlank()) {
                            defect(INCORRECT_SSP)
                        } else null
                    })
                    .check(doesNotDuplicateSspToken(pageBkData), When.isValid())
            } else {
                if (pageBkData.hasSSPID() || pageBkData.sspPageTokenList.isNotEmpty()) {
                    vb.check { defect(EXCESS_SSP_PARAMS) }
                }
            }

            // TODO move to before apply
            if (mode == OperationMode.EDIT
                && pageBkData.pageID in (1..1000) && pageBkData.pageID !in notInternalPages
            ) {
                vb.item(pageBkData.pageID, "PageID")
                    .check { defect(UNMODIFIABLE_PAGE, pageBkData.pageID) }
            }

            if (pageBkData.hasProductType() && pageBkData.productType !in knownProductTypes) {
                vb.item(pageBkData.productType, "ProductType")
                    .check { defect(WRONG_PRODUCT_TYPE, pageBkData.productType) }
            }

            if (pageBkData.hasCPA() && pageBkData.cpa !in (0..100)) {
                vb.item(pageBkData.cpa, "CPA")
                    .check { defect(INVALID_CPA, pageBkData.cpa) }
            }

            vb.list(pageBkData.placesList, "Places")
                .checkEach(Constraint { sp ->
                    if (!sp.hasStripeType() || !sp.hasStripeAnimation()) {
                        defect(STRIPE_NOT_DEFINED, sp.placeID)
                    } else null
                    // values are correct enum/boolean whenever parsed
                }, When.valueIs { it.placeID == CoreConstants.STRIPE_PLACE_ID })

            val reloadTimeout = pageBkData.optionsList.find { it.key == "ReloadTimeout" }
            if (reloadTimeout != null && (!reloadTimeout.hasValue() || reloadTimeout.value == null
                    || !Pattern.matches("^\\d+\$", reloadTimeout.value)
                    || reloadTimeout.value.toInt() !in (1..30))
            ) {
                vb.item(pageBkData.optionsList, "Options")
                    .item(reloadTimeout, "ReloadTimeout")
                    .check { defect(INVALID_RELOAD_TIMEOUT) }
            }

            if (pageBkData.hasLang()) {
                vb.item(pageBkData.lang, "Lang")
                    .check(inSet(knownLangs), defect(INCORRECT_LANG))
            }

            if (pageBkData.hasStore()) {
                vb.item(pageBkData.store, "Store")
                    .check(inSet(knownStores), defect(INCORRECT_STORE))
            }

            vb.list(pageBkData.slotsList, "Slots")
                .checkEachBy(slotValidatorProvider.validator())

            val blockBkDataValidator =
                blockBkDataValidatorProvider.bkDataValidator(pageBkData.pageID, null, allDsps)

            // TODO switch blocks by model (unused field would be empty, though validation is correct)
            vb.list(pageBkData.rtbBlocksList, "RtbBlocks")
                .checkEachBy(blockBkDataValidator)
                .checkEachBy(blockBkDataValidatorByPage(pageBkData))
            vb.list(pageBkData.directBlocksList, "DirectBlocks")
                .checkEachBy(blockBkDataValidator)
                .checkEachBy(blockBkDataValidatorByPage(pageBkData))

            vb.result
        }

    private fun blockBkDataValidatorByPage(pageBkData: TPartnerPage) =
        { blockBkData: TPartnerPage.TBlock ->
            val blockVb = ItemValidationBuilder.of<TPartnerPage.TBlock, Defect<Any>>(blockBkData)
            checkVideoDspTypes(
                blockBkDataValidatorProvider.getPublicId(blockBkData, pageBkData.pageID),
                pageBkData,
                blockBkData,
                blockVb
            )
            blockVb.result
        }

    private fun doesNotDuplicateSspToken(pageBkData: TPartnerPage):
        Constraint<MutableList<String>, Defect<Any>> =
        Constraint { tokens ->
            val sspLinkModel = pageToSspLinkModels[pageBkData.productID]!!

            if (sspLinkModel.tokensField == null) {
                return@Constraint null
            }

            val pn = sspLinkModel.pageIdField
            val duplicateSspPages = dslContext.select(pn)
                .from(sspLinkModel.table)
                .where(
                    sspLinkModel.sellerIdField.eq(pageBkData.sspid)
                        .and(pn.isNotNull)
                        .and(pn.notEqual(pageBkData.pageID))
                        .and(
                            tokens.map {
                                DSL.condition(
                                    "JSON_CONTAINS({0}, '\"{1}\"', '$')",
                                    sspLinkModel.tokensField,
                                    DSL.inline(it)
                                )
                            }.reduce(DSL::or)
                        )
                ).fetch(pn)

            if (duplicateSspPages.isNotEmpty()) {
                defect(DUPLICATE_SSP_TOKEN, duplicateSspPages)
            } else null
        }

    private val modelsSupportVideoWithoutRtbVideo = setOf(
        BlockType.INDOOR_BLOCK.literal,
        BlockType.OUTDOOR_BLOCK.literal
    )

    private fun checkVideoDspTypes(
        publicId: String,
        pageBkData: TPartnerPage,
        blockBkData: TPartnerPage.TBlock,
        bkDataVb: ItemValidationBuilder<TPartnerPage.TBlock, Defect<Any>>
    ) {
        if (blockBkData.hasDSPType()
            && blockBkData.dspType == CoreConstants.DspTypes.DSP_VIDEO.bkId()
            && blockBkData.hasVideo()
            && blockBkData.blockModel !in modelsSupportVideoWithoutRtbVideo
        ) {
            if (pageBkData.hasRtbVideo()) {
                if (CoreConstants.VideoBlockTypes.of(blockBkData.video.type) !in CoreConstants.VideoBlockTypes.values()) {
                    bkDataVb.item(blockBkData.video, "Video")
                        .check { defect(WRONG_VIDEO_TYPE) }
                }
            } else {
                bkDataVb.item(blockBkData.video, "Video")
                    .check { defect(PAGE_VIDEO_NOT_SUPPORTED, publicId) }
            }
        }
    }

    data class SspLinkModel(
        val pageModel: String,
        val table: Table<*>,
        val sellerIdField: Field<Long>,
        val pageIdField: Field<Long>,
        val tokensField: Field<*>?
    )

    private val pageToSspLinkModels = listOf(
        SspLinkModel(
            "ssp_video_an_site",
            PARTNER.SSP_LINK_VIDEO_APP,
            PARTNER.SSP_LINK_VIDEO_APP.SELLER_ID,
            PARTNER.SSP_LINK_VIDEO_APP.VIDEO_APP_ID,
            PARTNER.SSP_LINK_VIDEO_APP.TOKENS
        ),
        SspLinkModel(
            "ssp_mobile_app_settings",
            PARTNER.SSP_LINK_MOBILE_APP,
            PARTNER.SSP_LINK_MOBILE_APP.SELLER_ID,
            PARTNER.SSP_LINK_MOBILE_APP.APPLICATION_ID,
            PARTNER.SSP_LINK_MOBILE_APP.TOKENS,
        ),
        SspLinkModel(
            "ssp_context_on_site_campaign",
            PARTNER.SSP_LINK_CONTEXT_RTB,
            PARTNER.SSP_LINK_CONTEXT_RTB.SELLER_ID,
            PARTNER.SSP_LINK_CONTEXT_RTB.CONTEXT_RTB_ID,
            PARTNER.SSP_LINK_CONTEXT_RTB.TOKENS
        ),
    ).associateBy { it.pageModel }
}
