package ru.yandex.partner.core.entity.page.type.owner;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.CoreConstants;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.page.container.PageOperationContainer;
import ru.yandex.partner.core.entity.page.model.BaseInternalPage;
import ru.yandex.partner.core.entity.page.model.PageWithOwner;
import ru.yandex.partner.core.entity.page.repository.type.AbstractPageRepositoryTypeSupportWithMapper;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.dbschema.partner.tables.records.PagesRecord;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.partner.core.utils.FunctionalUtils.mapList;
import static ru.yandex.partner.dbschema.partner.tables.Pages.PAGES;

@ParametersAreNonnullByDefault
@Component
public class PageWithOwnerRepositoryTypeSupport
        extends AbstractPageRepositoryTypeSupportWithMapper<PageWithOwner, PagesRecord, PageOperationContainer> {

    private final UserService userService;
    private final JooqMapper<PageWithOwner> mapper;
    private final Set<ModelProperty<?, ?>> userModelProperties;

    protected PageWithOwnerRepositoryTypeSupport(DSLContext dslContext, UserService userService) {
        super(dslContext);
        this.userService = userService;

        this.mapper = JooqMapperBuilder.<PageWithOwner>builder()
                .map(property(PageWithOwner.OWNER_ID, PAGES.OWNER_ID))
                .readProperty(PageWithOwner.OWNER,
                        fromField(PAGES.OWNER_ID).by(value -> {
                            // Заглушка
                            // Для классов PartnerRepositoryTypeSupportWithMapper метод getAffectedModelProperties
                            // служит (для возврощаемых полей):
                            //  - для проверки, что у всех полей есть Reader (JooqReader.getFieldsToRead)
                            //  - для определения, при запросе каких полей необходимо вызывать этот тайпсуппорт
                            return null;
                        }))
                .build();

        this.userModelProperties = Set.of(
                User.ID,
                User.LOGIN,
                User.EMAIL,
                User.CLIENT_ID,
                User.ROLES,
                User.NAME,
                User.LASTNAME,
                User.MIDNAME,
                User.BUSINESS_UNIT,
                User.FEATURES,
                User.IS_ASSESSOR,
                User.IS_TUTBY,
                User.HAS_TUTBY_AGREEMENT,
                User.CURRENT_CURRENCY,
                User.ADFOX_INFOS
        );
    }

    @Override
    public JooqMapper<PageWithOwner> getJooqMapper() {
        return mapper;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<PageWithOwner> models) {
        // TODO: 22.03.2022 ваншотнуть pages: проставить внутренним owner_id
        // триггер айдишку не проставляет, так как в исходной таблице поля нет
        models.stream()
                .findFirst()
                .map(PageWithOwner::getClass)
                .map(BaseInternalPage.class::isAssignableFrom)
                .ifPresent(isInternal -> {
                    if (isInternal) {
                        models.forEach(
                                m -> m.setOwnerId(CoreConstants.ADINSIDE_USER_ID)
                        );
                    }
                });

        Map<Long, User> owners = userService.findAll(
                        QueryOpts.forClass(User.class)
                                .withFilter(CoreFilterNode.create(
                                        UserFilters.ID,
                                        FilterOperator.IN,
                                        mapList(models, PageWithOwner::getOwnerId)))
                                .withProps(userModelProperties)
                ).stream()
                .collect(Collectors.toMap(User::getId, Function.identity()));

        for (PageWithOwner model : models) {
            model.setOwner(owners.get(model.getOwnerId()));
        }
    }

    @Override
    public Class<PageWithOwner> getTypeClass() {
        return PageWithOwner.class;
    }

}
