package ru.yandex.partner.core.entity.page.type.site;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Sets;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.page.model.PageWithSite;
import ru.yandex.partner.core.entity.page.repository.type.AbstractPageRepositoryTypeSupport;
import ru.yandex.partner.core.entity.page.repository.type.PageRepositoryTypeSupportWithMapper;
import ru.yandex.partner.dbschema.partner.tables.Pages;

import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.partner.dbschema.partner.Tables.PAGES;

@ParametersAreNonnullByDefault
public abstract class PageWithSiteRepositoryTypeSupport<P extends PageWithSite, SR extends Record, PC>
        extends AbstractPageRepositoryTypeSupport<P, PC>
        implements PageRepositoryTypeSupportWithMapper<P, PC> {
    private final JooqMapper<P> mapper;

    protected PageWithSiteRepositoryTypeSupport(DSLContext dslContext) {
        super(dslContext);
        this.mapper = JooqMapperBuilder.<P>builder()
                .map(property(PageWithSite.DOMAIN_ID, PAGES.DOMAIN_ID))
                .readProperty(PageWithSite.DOMAIN,
                        fromField(Pages.PAGES.DOMAIN_ID).by(value -> {
                            // Заглушка
                            // Для классов PartnerRepositoryTypeSupportWithMapper метод getAffectedModelProperties
                            // служит (для возврощаемых полей):
                            //  - для проверки, что у всех полей есть Reader (JooqReader.getFieldsToRead)
                            //  - для определения, при запросе каких полей необходимо вызывать этот тайпсуппорт
                            return null;
                        }))
                .build();
    }

    @Override
    public JooqMapper<P> getJooqMapper() {
        return mapper;
    }


    protected abstract TableField<SR, Long> getSiteIdTableField();

    protected abstract JooqMapper<P> getSiteMapper();

    protected abstract Table<SR> getSiteTable();

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<P> models) {
        List<Long> domainIds = models.stream()
                .map(PageWithSite::getDomainId)
                .toList();

        Map<Long, SR> siteIdToDomainRecord = dslContext.selectFrom(getSiteTable())
                .where(getSiteIdTableField().in(domainIds))
                .fetch()
                .stream()
                .collect(Collectors.toMap(
                        r -> r.get(getSiteIdTableField()),
                        r -> r
                ));

        models.forEach(m -> {
            var siteRecord = siteIdToDomainRecord.get(m.getDomainId());
            if (siteRecord != null) {
                getSiteMapper().fromDb(siteRecord, m);
            }
        });
    }

    @Override
    public Set<ModelProperty<? super P, ?>> getAffectedModelProperties() {
        return Sets.union(
                PageRepositoryTypeSupportWithMapper.super.getAffectedModelProperties(),
                getSiteMapper().getReadableModelProperties()
        );
    }
}
