package ru.yandex.partner.core.entity.page.type.store;

import java.util.Collection;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;
import org.jooq.DSLContext;
import org.jooq.Table;
import org.jooq.TableField;

import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.page.container.PageOperationContainer;
import ru.yandex.partner.core.entity.page.model.PageWithStore;
import ru.yandex.partner.core.entity.page.repository.type.AbstractPageRepositoryTypeSupport;
import ru.yandex.partner.core.entity.page.repository.type.PageRepositoryTypeSupportWithMapper;

public abstract class BasePageWithStoreRepositoryTypeSupport<P extends PageWithStore> extends
        AbstractPageRepositoryTypeSupport<P, PageOperationContainer>
        implements PageRepositoryTypeSupportWithMapper<P, PageOperationContainer> {
    private final JooqMapper<P> mapper;
    private final JooqMapper<P> appMapper;

    public BasePageWithStoreRepositoryTypeSupport(
            DSLContext dslContext,
            JooqMapper<P> mapper,
            JooqMapper<P> appMapper
    ) {
        super(dslContext);
        this.mapper = mapper;
        this.appMapper = appMapper;
    }

    @Override
    public JooqMapper<P> getJooqMapper() {
        return mapper;
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<P> models) {
        var appIds = models.stream()
                .map(P::getApplicationId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        var appIdToAppRecord = dslContext
                .select(appMapper.getFieldsToRead())
                .from(getDomainTable())
                .where(getDomainIdField().in(appIds))
                .fetchMap(getDomainIdField());

        models.forEach(model -> {
            if (model.getApplicationId() != null) {
                var appRecord = appIdToAppRecord.get(model.getApplicationId());
                if (appRecord != null) {
                    appMapper.fromDb(appRecord, model);
                }
            }
        });
    }

    @NotNull
    protected abstract TableField<?, Long> getDomainIdField();

    protected abstract Table<?> getDomainTable();

    @Override
    public Set<ModelProperty<? super P, ?>> getAffectedModelProperties() {
        return Stream.of(
                getJooqMapper().getReadableModelProperties(),
                getJooqMapper().getWritableModelProperties(),
                appMapper.getReadableModelProperties(),
                appMapper.getWritableModelProperties()
        ).flatMap(Collection::stream).collect(Collectors.toSet());
    }
}
