package ru.yandex.partner.core.entity.queue.multistate;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.springframework.stereotype.Service;

import ru.yandex.partner.core.entity.queue.actions.TaskActionsEnum;
import ru.yandex.partner.core.messages.TaskActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.queue.TaskMultistate;
import ru.yandex.partner.core.multistate.queue.TaskStateFlag;
import ru.yandex.partner.core.queue.TaskData;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static ru.yandex.partner.core.multistate.queue.TaskStateFlag.CANCELED;
import static ru.yandex.partner.core.multistate.queue.TaskStateFlag.FINISHED_WITH_ERROR;
import static ru.yandex.partner.core.multistate.queue.TaskStateFlag.FINISHED_WITH_SUCCESS;
import static ru.yandex.partner.core.multistate.queue.TaskStateFlag.NEED_RESTART;
import static ru.yandex.partner.core.multistate.queue.TaskStateFlag.WORKING;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Service
public class TaskMultistateGraph extends AbstractMultistateGraph<TaskData, TaskStateFlag> {
    @Override
    public Multistate<TaskStateFlag> getMultistateFromModel(TaskData model) {
        return model.getMultistate();
    }

    @Override
    protected Multistate<TaskStateFlag> getMultistateForValue(Long multistateValue) {
        return new TaskMultistate(multistateValue);
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<TaskData, TaskStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<TaskData, TaskStateFlag>> actionEntryMap = new HashMap<>();

        actionEntryMap.put(TaskActionsEnum.START,
                getActionEntryBuilder(TaskActionMsg.START)
                        .setPredicate(empty())
                        .setSetFlags(Map.of(
                                WORKING, true
                        ))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.PASS_MILESTONE,
                getActionEntryBuilder(TaskActionMsg.PASS_MILESTONE)
                        .setPredicate(has(WORKING))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.CANCEL,
                getActionEntryBuilder(TaskActionMsg.CANCEL)
                        .setPredicate(empty())
                        .setSetFlags(Map.of(
                                CANCELED, true
                        ))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.FINISH_WITH_SUCCESS,
                getActionEntryBuilder(TaskActionMsg.FINISH_WITH_SUCCESS)
                        .setPredicate(has(WORKING))
                        .setSetFlags(Map.of(
                                WORKING, false,
                                FINISHED_WITH_SUCCESS, true
                        ))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.FINISH_WITH_ERROR,
                getActionEntryBuilder(TaskActionMsg.FINISH_WITH_ERROR)
                        .setPredicate(has(WORKING))
                        .setSetFlags(Map.of(
                                WORKING, false,
                                FINISHED_WITH_ERROR, true
                        ))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.RESTART,
                getActionEntryBuilder(TaskActionMsg.RESTART)
                        .setPredicate(has(FINISHED_WITH_ERROR).or(has(NEED_RESTART)))
                        .setSetFlags(Map.of(
                                FINISHED_WITH_ERROR, false,
                                NEED_RESTART, false
                        ))
                        .build()
        );

        actionEntryMap.put(TaskActionsEnum.NEED_RESTART,
                getActionEntryBuilder(TaskActionMsg.NEED_RESTART)
                        .setPredicate(has(WORKING))
                        .setSetFlags(Map.of(
                                WORKING, false,
                                NEED_RESTART, true))
                        .build()
        );

        return actionEntryMap;
    }

    private ActionEntry.Builder<TaskData, TaskStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return new ActionEntry.Builder<>(titleMsg);
    }

    @Override
    public Class<TaskData> getModelClass() {
        return TaskData.class;
    }

    @Override
    public Multistate<TaskStateFlag> convertMultistate(List<TaskStateFlag> enabledFlags) {
        return new TaskMultistate(enabledFlags);
    }
}
