package ru.yandex.partner.core.entity.queue.service;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.partner.core.queue.Task;
import ru.yandex.partner.core.queue.TaskData;
import ru.yandex.partner.core.queue.TaskFactory;

@Service
public class TaskFacade {

    private final Map<Class<?>, TaskFactory<?>> taskFactoryMap;

    @Autowired
    public TaskFacade(List<TaskFactory<?>> taskFactories) {
        this.taskFactoryMap = taskFactories.stream()
                .collect(Collectors.toMap(TaskFactory::getTaskClass, Function.identity()));
    }

    public <T extends Task<?, ?>> T fromTaskData(TaskData taskData, Class<T> clazz) {
        TaskFactory<?> taskFactory = taskFactoryMap.get(clazz);

        if (taskFactory != null) {
            try {
                return clazz.cast(taskFactory.fromTaskData(taskData));
            } catch (ClassCastException exception) {
                throw new IllegalStateException("Task class is mapped to a wrong factory", exception);
            }
        } else {
            throw new IllegalStateException("No task factory found for task " + clazz.getSimpleName());
        }
    }

    public TaskFactory<?> getFactoryForTask(Task<?, ?> task) {
        return getFactoryForTaskClass(task.getClass());
    }

    public <T extends Task<?, ?>> TaskFactory<?> getFactoryForTaskClass(Class<T> clazz) {
        return Optional.ofNullable(taskFactoryMap.get(clazz))
                .orElseThrow(
                        () -> new IllegalStateException("No task factory found for class " + clazz.getSimpleName())
                );
    }
}
