package ru.yandex.partner.core.entity.simplemodels;

import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.ModelService;
import ru.yandex.partner.core.entity.Query;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.UpdateLifecycle;
import ru.yandex.partner.core.filter.CoreFilterNode;

import static java.util.function.Function.identity;
import static ru.yandex.direct.multitype.typesupport.TypeFilteringUtils.filterModelsOfClass;

/**
 * Базовый класс для работ с простыми моделями.
 */
@ParametersAreNonnullByDefault
public abstract class AbstractSimpleService<M extends ModelWithId> implements ModelService<M> {
    private final Class<M> modelClass;
    private final SimpleRepository<M> repository;
    private final SimpleValidatorService<M> validatorService;

    public AbstractSimpleService(Class<M> modelClass,
                                 SimpleRepository<M> repository,
                                 SimpleValidatorService<M> validatorService) {
        this.modelClass = modelClass;
        this.repository = repository;
        this.validatorService = validatorService;
    }

    @Override
    public ActionsResult<?> create(List<M> models, IncomingFields updatedFields) {
        ActionsResult<?> actionsResult = new ActionsResult<>();
        MassResult<M> massResult = createOrUpdate(models);
        actionsResult.putToResults(modelClass, massResult);
        actionsResult.setCommitted(massResult.isSuccessful());
        return actionsResult;
    }

    @Override
    public ActionsResult<?> update(List<M> models, List<M> prefetchedModels, UpdateLifecycle<M> lifecycle) {
        var modelsFromDb = prefetchedModels.stream().collect(Collectors.toMap(ModelWithId::getId, identity()));
        models.forEach(model -> lifecycle.beforeUpdate(modelsFromDb.get(model.getId()), model));
        var res = new ActionsResult<>();
        res.putToResults(getModelClass(), createOrUpdate(models));
        return res;
    }

    public MassResult<M> createOrUpdate(List<M> entities) {
        ValidationResult<List<M>, Defect> validationResult = validatorService.validate(entities);
        if (validationResult.hasAnyErrors()) {
            return MassResult.brokenMassAction(entities, validationResult);
        } else {
            List<M> afterUpdate = repository.createOrUpdate(entities);
            return MassResult.successfulMassAction(afterUpdate, ValidationResult.success(afterUpdate));
        }
    }

    @Override
    public <S extends M> List<S> findAll(Query<S> queryOpts) {
        Class<S> subType = queryOpts.getClazz();
        if (subType == getModelClass()) {
            QueryOpts<S> opts = (QueryOpts<S>) queryOpts;
            return filterModelsOfClass(
                    repository.getAll(
                            (CoreFilterNode<M>) opts.getFilter(),
                            opts.getLimitOffset(),
                            opts.getOrderByList(),
                            opts.isForUpdate()
                    ),
                    subType
            );
        }
        throw new UnsupportedOperationException("Subtyping not supported for KvStoreFrontend");
    }

    @Override
    public <S extends M> long count(Query<S> opts) {
        return repository.getCountByCondition((CoreFilterNode<M>) opts.getFilter());
    }

    protected abstract Class<M> getModelClass();
}
