package ru.yandex.partner.core.entity.simplemodels.kvstorefrontend.repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.impl.TableImpl;
import org.springframework.stereotype.Repository;

import ru.yandex.partner.core.entity.kvstorefrontend.model.KvStoreFrontend;
import ru.yandex.partner.core.entity.simplemodels.AbstractSimpleRepository;
import ru.yandex.partner.core.entity.simplemodels.kvstorefrontend.filter.KvStoreFrontendModelContainerFilters;
import ru.yandex.partner.core.entity.utils.ConditionUtils;
import ru.yandex.partner.core.entity.utils.DSLUtils;
import ru.yandex.partner.dbschema.partner.tables.records.KvStoreFrontendRecord;
import ru.yandex.partner.defaultconfiguration.PartnerLocalDateTime;

import static ru.yandex.partner.dbschema.partner.Tables.KV_STORE_FRONTEND;

@ParametersAreNonnullByDefault
@Repository
public class KvStoreFrontendRepository extends AbstractSimpleRepository<KvStoreFrontend, KvStoreFrontendRecord> {
    private final Map<String, Field<?>> dbFields;

    public KvStoreFrontendRepository(DSLContext dslContext,
                                     KvStoreFrontendModelContainerFilters modelContainerFilters) {
        super(dslContext, modelContainerFilters);
        this.dbFields = prepareDbFieldsMapping();
    }

    @Override
    public List<KvStoreFrontend> createOrUpdate(List<KvStoreFrontend> entities) {
        var now = PartnerLocalDateTime.now();
        var query = getDslContext().insertInto(
                KV_STORE_FRONTEND,
                KV_STORE_FRONTEND.USER_ID,
                KV_STORE_FRONTEND.KEY,
                KV_STORE_FRONTEND.VALUE,
                KV_STORE_FRONTEND.CREATE_TIME,
                KV_STORE_FRONTEND.UPDATE_TIME
        );

        for (var entity : entities) {
            query.values(
                    entity.getUserId(),
                    entity.getKey(),
                    entity.getValue(),
                    now,
                    now
            );
        }

        query.onDuplicateKeyUpdate()
                .set(KV_STORE_FRONTEND.VALUE, DSLUtils.getValuesStatement(KV_STORE_FRONTEND.VALUE))
                .set(KV_STORE_FRONTEND.UPDATE_TIME, DSLUtils.getValuesStatement(KV_STORE_FRONTEND.UPDATE_TIME));

        query.execute();

        var selectQuery = getBaseSelectQuery();
        selectQuery.addConditions(ConditionUtils.toMapCondition(
                StreamEx.of(entities).groupingBy(
                        KvStoreFrontend::getUserId, Collectors.mapping(KvStoreFrontend::getKey, Collectors.toList())),
                KV_STORE_FRONTEND.USER_ID, KV_STORE_FRONTEND.KEY));

        return getModels(selectQuery);
    }

    @Override
    public int insert(List<KvStoreFrontend> entities) {
        return createOrUpdate(entities).size();
    }

    @Override
    public int update(List<KvStoreFrontend> entities) {
        return createOrUpdate(entities).size();
    }

    @Override
    public void delete(Collection<KvStoreFrontend> currentModels) {
        throw new UnsupportedOperationException("Метод не реализован");
    }

    @Override
    protected TableImpl<KvStoreFrontendRecord> getTable() {
        return KV_STORE_FRONTEND;
    }

    @Override
    protected Function<Record, KvStoreFrontend> fetchQueryFunction() {
        return record -> new KvStoreFrontend()
                .withId(record.get(KV_STORE_FRONTEND.ID))
                .withUserId(record.get(KV_STORE_FRONTEND.USER_ID))
                .withKey(record.get(KV_STORE_FRONTEND.KEY))
                .withValue(record.get(KV_STORE_FRONTEND.VALUE))
                .withCreateTime(record.get(KV_STORE_FRONTEND.CREATE_TIME))
                .withUpdateTime(record.get(KV_STORE_FRONTEND.UPDATE_TIME));
    }

    @Override
    public Map<String, Field<?>> getDbFields() {
        return dbFields;
    }

    @Override
    protected Field<?> getIdField() {
        return KV_STORE_FRONTEND.ID;
    }

    @Override
    protected Class<KvStoreFrontend> getModelClass() {
        return KvStoreFrontend.class;
    }

    protected Map<String, Field<?>> prepareDbFieldsMapping() {
        Map<String, Field<?>> data = new HashMap<>();
        data.put(KvStoreFrontend.ID.name(), KV_STORE_FRONTEND.ID);
        data.put(KvStoreFrontend.USER_ID.name(), KV_STORE_FRONTEND.USER_ID);
        data.put(KvStoreFrontend.KEY.name(), KV_STORE_FRONTEND.KEY);
        data.put(KvStoreFrontend.VALUE.name(), KV_STORE_FRONTEND.VALUE);
        data.put(KvStoreFrontend.CREATE_TIME.name(), KV_STORE_FRONTEND.CREATE_TIME);
        data.put(KvStoreFrontend.UPDATE_TIME.name(), KV_STORE_FRONTEND.UPDATE_TIME);
        return data;
    }
}
