package ru.yandex.partner.core.entity.simplemodels.kvstorefrontend.service.validation;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.kvstorefrontend.model.KvStoreFrontend;
import ru.yandex.partner.core.entity.simplemodels.SimpleRepository;
import ru.yandex.partner.core.entity.simplemodels.SimpleValidatorService;
import ru.yandex.partner.core.entity.simplemodels.kvstorefrontend.filter.KvStoreFrontendFilters;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.operator.FilterOperator;

import static java.util.stream.Collectors.toUnmodifiableSet;

@Service
@ParametersAreNonnullByDefault
public class KvStoreFrontendValidatorService implements SimpleValidatorService<KvStoreFrontend> {
    private final Integer maxSize;
    private final SimpleRepository<KvStoreFrontend> repository;

    @Autowired
    public KvStoreFrontendValidatorService(
            @Value("${kvStore.maxSize:1000}") Integer maxSize,
            SimpleRepository<KvStoreFrontend> repository
    ) {
        this.maxSize = maxSize;
        this.repository = repository;
    }

    @Override
    public ValidationResult<List<KvStoreFrontend>, Defect> validate(List<KvStoreFrontend> entities) {
        return ListValidationBuilder.of(entities, Defect.class)
                .checkEach(insertionDoesNotExceedMaxCount(entities))
                .getResult();
    }

    private Constraint<KvStoreFrontend, Defect> insertionDoesNotExceedMaxCount(List<KvStoreFrontend> entities) {
        var userIds = StreamEx.of(entities)
                .map(KvStoreFrontend::getUserId)
                .collect(toUnmodifiableSet());

        CoreFilterNode<KvStoreFrontend> filterByUsers = filterByUsers(userIds);
        Map<Long, Integer> kvCountPerUser = repository.getCountGroupedBy(
                KvStoreFrontend.USER_ID.name(),
                filterByUsers
        );
        return insertionDoesNotExceedMaxCount(
                kvCountPerUser,
                repository.existingIds(filterByUsers),
                maxSize
        );
    }

    @NotNull
    private CoreFilterNode<KvStoreFrontend> filterByUsers(Set<Long> userIds) {
        return CoreFilterNode.create(KvStoreFrontendFilters.USER_ID, FilterOperator.IN, userIds);
    }

    private Constraint<KvStoreFrontend, Defect> insertionDoesNotExceedMaxCount(
            Map<Long, Integer> kvCountPerUser,
            Set<Long> existingIds,
            Integer maxCount
    ) {
        Map<Long, Integer> mutableCountPerUser = new HashMap<>(kvCountPerUser);

        return item -> {
            long userId = item.getUserId();

            // no insertion would be performed for entry, update doesn't increase count
            if (item.getId() != null && existingIds.contains(item.getId())) {
                return null;
            }

            int sizeAfterInsert = mutableCountPerUser.compute(userId, (k, v) -> v == null ? 1 : v + 1);

            return sizeAfterInsert > maxCount
                    ? KvStoreDefect.userStoreOverflow(userId)
                    : null;
        };
    }
}
