package ru.yandex.partner.core.entity.tasks;

import java.io.IOException;
import java.time.Duration;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.user.filter.UserFilters;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.queue.AbstractTask;
import ru.yandex.partner.core.queue.TaskData;
import ru.yandex.partner.core.queue.TaskPayload;
import ru.yandex.partner.core.queue.TaskType;

public class EmailToProcessingTask extends AbstractTask<EmailToProcessingTask.Payload, Void> {

    private final UserService userService;

    public EmailToProcessingTask(
            Payload payload,
            TaskData savedTaskData,
            UserService userService
    ) {
        super(payload, savedTaskData);
        this.userService = userService;
    }

    @Override
    public Void execute() {
        User user = userService.findAll(
                QueryOpts.forClass(User.class)
                        .withFilter(CoreFilterNode.eq(UserFilters.ID, getPayload().userId))
                    .withProps(Set.of(User.ID))
        ).get(0);
        System.out.println("Sending email for user " + user.getId());
        // TODO: inject email service and send
        return null;
    }

    @Override
    public TaskData getTaskData() {
        return getSavedTaskData();
    }

    @Override
    public Duration getEstimatedTime() {
        return Duration.ofSeconds(60);
    }

    @JsonDeserialize(using = Payload.Deserializer.class)
    public static class Payload implements TaskPayload {

        @JsonProperty("uid")
        private final Long userId;

        public Payload(Long userId) {
            this.userId = userId;
        }

        public static Payload of(Long userId) {
            return new Payload(userId);
        }

        @Override
        public String serializeParams() {
            try {
                return new ObjectMapper().writeValueAsString(this);
            } catch (IOException exception) {
                throw new RuntimeException("Failed to serialize EmailToProcessingTask.Payload", exception);
            }
        }

        public Long getUserId() {
            return userId;
        }

        public Long setUserId() {
            return userId;
        }

        @Override
        public int getTypeId() {
            return TaskType.SEND_EMAIL_TO_PROCESSING.getTypeId();
        }

        @Override
        public Long getGroupId() {
            return null;
        }

        public static class Deserializer extends StdDeserializer<Payload> {

            public Deserializer() {
                this(null);
            }

            public Deserializer(Class<?> vc) {
                super(vc);
            }

            @Override
            public Payload deserialize(JsonParser jp, DeserializationContext ctxt)
                    throws IOException {
                JsonNode node = jp.getCodec().readTree(jp);
                JsonNode userIdNode = node.get("uid");
                if (userIdNode != null && userIdNode.isNumber()) {
                    return new Payload(node.get("uid").longValue());
                } else {
                    throw new JsonParseException(jp, "Expected numeric userId");
                }
            }
        }
    }
}
