package ru.yandex.partner.core.entity.tasks.convertcpm;

import java.math.BigDecimal;
import java.time.Duration;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.partner.core.entity.queue.exceptions.TaskExecutionException;
import ru.yandex.partner.core.entity.user.type.cpm.CpmConversion;
import ru.yandex.partner.core.entity.user.type.cpm.CpmConversionFactory;
import ru.yandex.partner.core.queue.InJavaSerializationStrategyKt;
import ru.yandex.partner.core.queue.Task;
import ru.yandex.partner.core.queue.TaskData;
import ru.yandex.partner.core.queue.TaskPayload;
import ru.yandex.partner.core.queue.TaskType;
import ru.yandex.partner.libs.annotation.PartnerTransactional;

public class ConvertCpmTask implements Task<ConvertCpmTask.Payload, Void> {
    private final ObjectMapper objectMapper;
    private final TaskData taskData;
    private final CpmConversionFactory cpmConversionFactory;

    public ConvertCpmTask(
            TaskData taskData,
            ObjectMapper objectMapper,
            CpmConversionFactory cpmConversionFactory
    ) {
        this.taskData = taskData;
        this.objectMapper = objectMapper;
        this.cpmConversionFactory = cpmConversionFactory;
    }


    @PartnerTransactional
    @Override
    public Void execute() {
        CpmConversion cpmConversion = cpmConversionFactory.fromPayload(getPayload(taskData.getParams()));
        cpmConversion.convert();
        return null;
    }

    private Payload getPayload(String params) {
        try {
            return objectMapper.readValue(params, ConvertCpmTask.Payload.class);
        } catch (JsonProcessingException e) {
            throw new TaskExecutionException(String.format("Deserialization failed for params: %s", params), e);
        }
    }


    @Override
    public TaskData getTaskData() {
        return taskData;
    }

    @Override
    public Duration getEstimatedTime() {
        return Duration.ofHours(1);
    }

    public record Payload(
            String currency,
            @JsonProperty("currency_rate") BigDecimal currencyRate,
            @JsonProperty("user") Long userId
    ) implements TaskPayload {

        @Override
        public String serializeParams() {
            return InJavaSerializationStrategyKt.serializeParams();
        }

        @Override
        public int getTypeId() {
            return TaskType.CONVERT_CPM_IN_USER_BLOCKS.getTypeId();
        }

        @Override
        public Long getGroupId() {
            return null;
        }
    }
}
