package ru.yandex.partner.core.entity.tasks.doaction;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.ModelWithId;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.ActionUserIdContext;
import ru.yandex.partner.core.action.exception.ActionError;
import ru.yandex.partner.core.action.factories.AllCustomPayloadActionsFactory;
import ru.yandex.partner.core.action.result.ActionsResult;
import ru.yandex.partner.core.entity.ModelQueryService;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.queue.exceptions.TaskExecutionException;
import ru.yandex.partner.core.filter.CoreFilterNode;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.queue.InJavaSerializationStrategyKt;
import ru.yandex.partner.core.queue.Task;
import ru.yandex.partner.core.queue.TaskData;
import ru.yandex.partner.core.queue.TaskPayload;
import ru.yandex.partner.core.queue.TaskType;
import ru.yandex.partner.core.service.entitymanager.EntityManager;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

import static ru.yandex.partner.core.filter.CoreFilterNode.eq;
import static ru.yandex.partner.core.filter.CoreFilterNode.neutral;

@ParametersAreNonnullByDefault
public class DoActionTask implements Task<DoActionTask.DoActionTaskPayload, ActionsResult<?>> {
    private static final Logger LOGGER = LoggerFactory.getLogger(DoActionTask.class);
    private final ActionPerformer actionPerformer;
    private final AllCustomPayloadActionsFactory factory;
    private final TaskData taskData;
    private final ObjectMapper objectMapper;
    private final EntityManager entityManager;
    private final ActionUserIdContext actionUserIdContext;

    public DoActionTask(
            ActionPerformer actionPerformer,
            AllCustomPayloadActionsFactory factory,
            ObjectMapper objectMapper,
            TaskData taskData,
            EntityManager entityManager,
            ActionUserIdContext actionUserIdContext) {
        this.actionPerformer = actionPerformer;
        this.factory = factory;
        this.taskData = taskData;
        this.objectMapper = objectMapper;
        this.entityManager = entityManager;
        this.actionUserIdContext = actionUserIdContext;
    }

    @Override
    public ActionsResult<?> execute() {
        DoActionTaskPayload payload = getPayload(taskData.getParams());
        String actionName = payload.getActionName();

        List<Long> ids = new ArrayList<>();
        String errorData = this.taskData.getErrorData();
        if (!Objects.isNull(errorData) && !errorData.isEmpty()) {
            LOGGER.info("ids are from errorData");

            try {
                ids.addAll(objectMapper.readValue(
                        errorData,
                        new TypeReference<List<Long>>() {
                        }
                ));
            } catch (JsonProcessingException e) {
                throw new TaskExecutionException(String.format("Deserialization failed for %s", errorData), e);
            }
        } else {
            LOGGER.info("ids are from payload");

            ids.addAll(this.getIdsByPayload(payload));
        }

        LOGGER.info(String.format("\"%d\" blocks found for action \"%s\": %s", ids.size(), actionName, ids));

        var action = factory.getFactory(actionName, entityManager.getEntityClass(payload.getModelName()))
                .createAction(ids, null);

        //Работает под пользователем указанным в таске
        actionUserIdContext.setUserId(payload.getUserId());

        var result = actionPerformer.doActions(false, action);
        var errorIds = Lists.<Long>newArrayList();
        result.getErrors().values().forEach(it -> it.forEach((id, value) -> {
            var defects = StreamEx.of(value)
                    .map(ActionError::getDefectType)
                    .partitioningBy(defectType -> defectType.equals(ActionError.ActionDefectType.CAN_NOT_DO_ACTION));
            if (!defects.get(true).isEmpty()) {
                //Если нельзя выполнить  экшен то просто логгируем
                LOGGER.info(String.format("Can not do action \"%s\" with id: %d ", actionName, id));
            }
            if (!defects.get(false).isEmpty()) {
                errorIds.add(id);
            }
        }));
        actionUserIdContext.setDefault();

        if (!errorIds.isEmpty()) {
            throw new TaskExecutionException(String.format("Couldn't apply the action \"%s\" to some entities",
                    actionName)) {
                @Override
                public Object getErrorData() {
                    return errorIds;
                }
            };
        }

        return result;
    }

    public DoActionTask.DoActionTaskPayload getPayload(String params) {
        DoActionTask.DoActionTaskPayload doActionTaskPayload;
        try {
            doActionTaskPayload = objectMapper.readValue(params, DoActionTask.DoActionTaskPayload.class);
        } catch (JsonProcessingException e) {
            throw new TaskExecutionException(String.format("Deserialization failed for %s", params), e);
        }

        return doActionTaskPayload;
    }

    public <T extends ModelWithId> List<Long> getIdsByPayload(DoActionTask.DoActionTaskPayload payload) {
        String actionName = payload.getActionName();

        ModelQueryService<T> modelQueryService =
                (ModelQueryService<T>) this.entityManager.getModelQueryService(payload.getModelName());

        CoreFilterNode<T> filter = neutral();

        if (!Objects.isNull(payload.getPageId())) {
            MetaFilter pageIdMetaFilter = modelQueryService.getMetaFilterForDoAction(DoActionFilterEnum.PAGE_ID);
            filter = CoreFilterNode.or(filter, eq(pageIdMetaFilter, payload.getPageId()));
        }

        //Выбираем только блоки для которых можно применить экшен
        MultistateGraph<? extends ModelWithId, ?> multistateGraph =
                this.entityManager.getMultistateGraph(payload.getModelName());
        List<Long> multistates = multistateGraph.getMultistatesByAction(actionName);

        MetaFilter multistateMetaFilter = modelQueryService.getMetaFilterForDoAction(DoActionFilterEnum.MULTISTATE);
        filter = CoreFilterNode.and(filter, CoreFilterNode.in(multistateMetaFilter, multistates));

        return modelQueryService.getIds(QueryOpts
                .forClass((Class<T>) entityManager.getEntityClass(payload.getModelName()))
                .withFilterByIds(payload.getModelIds())
                .withFilter(filter)
        );
    }

    @Override
    public TaskData getTaskData() {
        return this.taskData;
    }

    @Override
    public Duration getEstimatedTime() {
        return Duration.ofSeconds(60);
    }

    public static class DoActionTaskPayload implements TaskPayload {

        private List<Long> modelIds;
        private String actionName;
        private String modelName;
        private Long userId;
        private Long pageId;
        private Long vmapId;


        @Override
        public String serializeParams() {
            return InJavaSerializationStrategyKt.serializeParams();
        }

        @Override
        public int getTypeId() {
            return TaskType.DO_ACTION.getTypeId();
        }

        @Override
        public Long getGroupId() {
            return null;
        }

        public List<Long> getModelIds() {
            return modelIds;
        }

        public void setModelIds(List<Long> modelIds) {
            this.modelIds = modelIds;
        }

        public String getActionName() {
            return actionName;
        }

        public void setActionName(String actionName) {
            this.actionName = actionName;
        }

        public String getModelName() {
            return modelName;
        }

        public void setModelName(String modelName) {
            this.modelName = modelName;
        }

        public Long getUserId() {
            return userId;
        }

        public void setUserId(Long userId) {
            this.userId = userId;
        }

        public Long getPageId() {
            return pageId;
        }

        public void setPageId(Long pageId) {
            this.pageId = pageId;
        }

        public Long getVmapId() {
            return vmapId;
        }

        public void setVmapId(Long vmapId) {
            this.vmapId = vmapId;
        }
    }
}
