package ru.yandex.partner.core.entity.user.actions;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.partner.core.action.AbstractActionContextWithModelProperty;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.entity.IncomingFields;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.user.UserCopyMapper;
import ru.yandex.partner.core.entity.user.actions.log.UserActionsLogger;
import ru.yandex.partner.core.entity.user.model.BaseUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.UserService;
import ru.yandex.partner.core.entity.user.service.UserUpdateOperationFactory;
import ru.yandex.partner.core.filter.CoreFilterNode;

@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class UserActionContext extends AbstractActionContextWithModelProperty<User> {
    private final UserService userService;
    private final UserUpdateOperationFactory userUpdateOperationFactory;
    private final UserCopyMapper userCopyMapper;

    @Autowired
    public UserActionContext(UserService userService,
                             UserUpdateOperationFactory userUpdateOperationFactory,
                             UserActionsLogger userActionsLogger, UserCopyMapper userCopyMapper,
                             UserActionErrorHandler errorHandler) {
        super(userActionsLogger, errorHandler);
        this.userService = userService;
        this.userUpdateOperationFactory = userUpdateOperationFactory;
        this.userCopyMapper = userCopyMapper;
    }

    @Override
    public List<User> getModelsByIds(
            Collection<Long> ids, Set<ModelProperty<?, ?>> fields,
            CoreFilterNode<User> filter, boolean forUpdate) {
        return userService.findAll(
                QueryOpts.forClass(User.class)
                        .withFilterByIds(ids)
                        .withFilter(filter)
                        .withProps(fields)
                        .forUpdate(forUpdate)
        );
    }

    @Override
    public User clone(User item) {
        return userCopyMapper.copy(item);
    }

    @Override
    public ModelChanges<User> createModelChanges(Long id) {
        return new ModelChanges<>(id, User.class);
    }

    @Override
    public Operation<Long> getUpdateOperation(List<ModelChanges<User>> modelChanges, IncomingFields updatedFields,
                                              boolean rollbackIfErrors) {
        List<ActionModelContainerImpl<User>> containersWithoutErrors =
                this.getContainersWithoutErrors(modelChanges.stream().map(ModelChanges::getId)
                        .collect(Collectors.toList()));

        return userUpdateOperationFactory.createUpdateOperationWithPreloadedModels(
                rollbackIfErrors ? Applicability.FULL : Applicability.PARTIAL,
                modelChanges.stream().map(
                        userModelChanges -> userModelChanges.castModel(BaseUser.class)
                ).collect(Collectors.toList()),
                containersWithoutErrors.stream().map(ActionModelContainerImpl::getNonChangedItem)
                        .collect(Collectors.toList())
        );
    }

    @Override
    public Class<User> getEntityClass() {
        return User.class;
    }
}
