package ru.yandex.partner.core.entity.user.actions;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.action.ActionConfiguration;
import ru.yandex.partner.core.action.ActionContext;
import ru.yandex.partner.core.action.ActionModelContainerImpl;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.helper.ActionOptsHelper;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.multistate.user.UserStateFlag;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

public class UserActionEdit extends UserAction {

    private final Map<Long, ModelChanges<User>> modelChangesMap;
    private final ObjectMapper objectMapper;

    @SuppressWarnings("ParameterNumber")
    public UserActionEdit(ActionConfiguration<User, ?> parentFactory,
                          String actionName,
                          Collection<ModelChanges<User>> modelChangesMap,
                          MultistateGraph<User, UserStateFlag> multistateGraph,
                          ActionPerformer actionPerformer,
                          ObjectMapper objectMapper,
                          UserActionErrorHandler errorHandler) {
        this(
                parentFactory,
                actionName,
                modelChangesMap.stream().collect(Collectors.toMap(ModelChanges::getId, Function.identity())),
                multistateGraph,
                actionPerformer,
                objectMapper,
                errorHandler
        );
    }

    @SuppressWarnings("ParameterNumber")
    public UserActionEdit(ActionConfiguration<User, ?> parentFactory,
                          String actionName,
                          Map<Long, ModelChanges<User>> modelChangesMap,
                          MultistateGraph<User, UserStateFlag> multistateGraph,
                          ActionPerformer actionPerformer,
                          ObjectMapper objectMapper,
                          UserActionErrorHandler errorHandler) {
        super(
                parentFactory,
                actionName,
                modelChangesMap.keySet(),
                multistateGraph,
                actionPerformer,
                errorHandler
        );
        this.modelChangesMap = modelChangesMap;
        this.objectMapper = objectMapper;
    }

    @Override
    public Set<ModelProperty<?, ?>> getDependsOn() {
        var stream1 = super.getDependsOn().stream();
        var stream2 = modelChangesMap.values().stream().map(ModelChanges::getChangedPropsNames).flatMap(Set::stream);

        return Stream.concat(stream1, stream2).collect(Collectors.toSet());
    }

    @Override
    public String getSerializedOpts(Long id) {
        return ActionOptsHelper.prepareSerializedOptsForUpdated(this.modelChangesMap.getOrDefault(id, null),
                objectMapper, User.class);
    }

    @Override
    public void onAction(ActionContext<User, ActionModelContainerImpl<User>> ctx,
                         List<ActionModelContainerImpl<User>> containers) {
        for (ActionModelContainerImpl<User> container : containers) {
            ModelChanges<User> modelChanges = this.getModelChangesById(container.getProperty(User.ID));

            modelChanges.getChangedPropsNames().forEach(modelProperty -> container.changeProperty(
                    (ModelProperty<User, Object>) modelProperty,
                    modelChanges.getChangedProp(modelProperty)
            ));
        }
    }

    public ModelChanges<User> getModelChangesById(Long id) {
        return Objects.requireNonNull(modelChangesMap.get(id),
                "Edit action: No changes found for model: " + id);
    }
}
