package ru.yandex.partner.core.entity.user.multistate;

import java.io.IOException;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateDeserializer;
import com.fasterxml.jackson.datatype.jsr310.deser.LocalDateTimeDeserializer;
import com.google.common.annotations.VisibleForTesting;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.dbschema.partner.tables.Contracts;
import ru.yandex.partner.libs.extservice.balance.BalanceContractUtils;
import ru.yandex.partner.libs.extservice.balance.ContractLiveData;

import static ru.yandex.partner.core.utils.FunctionalUtils.listToSet;

@Component
public class CanChangeRequisitesCheck {
    private final DSLContext dslContext;
    private final ObjectMapper objectMapper;

    @Autowired
    public CanChangeRequisitesCheck(DSLContext dslContext, ObjectMapper objectMapper) {
        this.dslContext = dslContext;
        this.objectMapper = objectMapper;
    }


    public Set<ModelProperty<? super User, ?>> getRequiredProperties() {
        return Set.of(User.CLIENT_ID);
    }

    public List<Boolean> check(List<? extends User> users) {
        var clientIds = listToSet(users, User::getClientId);
        Map<Long, List<String>> contractsMap = dslContext
                .select(Contracts.CONTRACTS.CLIENT_ID, Contracts.CONTRACTS.CONTRACT)
                .from(Contracts.CONTRACTS)
                .where(Contracts.CONTRACTS.CLIENT_ID.in(clientIds)
                        .and(Contracts.CONTRACTS.PERSON_ID.isNotNull())
                        .and(Contracts.CONTRACTS.PERSON_ID.ne(0L))
                )
                .orderBy(Contracts.CONTRACTS.ID.desc())
                .fetchGroups(Contracts.CONTRACTS.CLIENT_ID, Contracts.CONTRACTS.CONTRACT);


        return users.stream()
                .map(user -> contractsMap.get(user.getClientId()))
                .map(this::hasLiveContract)
                .toList();
    }

    @VisibleForTesting
    boolean hasLiveContract(List<String> contractStrings) {
        if (contractStrings == null) {
            return false;
        }

        boolean hasLiveContract = false;
        LocalDate today = LocalDate.now();
        Iterator<String> contractStringsIterator = contractStrings.iterator();
        while (!hasLiveContract && contractStringsIterator.hasNext()) {
            try {
                String contractString = contractStringsIterator.next();
                ContractLiveDataDto contractLiveDataDto =
                        objectMapper.readValue(contractString, ContractLiveDataDto.class);
                hasLiveContract = BalanceContractUtils.isLiveContract(today, contractLiveDataDto);
            } catch (JsonProcessingException e) {
                throw new RuntimeException("Error during deserialize from json", e);
            }
        }

        return hasLiveContract;
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static class ContractLiveDataDto implements ContractLiveData {
        @JsonDeserialize(using = CustomDeserialize.class)
        private LocalDate dt;
        @JsonProperty("end_dt")
        @JsonDeserialize(using = CustomDeserialize.class)
        private LocalDate contractEndDt;

        @JsonProperty("is_signed")
        @JsonDeserialize(using = CustomDeserialize.class)
        private LocalDate isSigned;

        @JsonProperty("is_faxed")
        @JsonDeserialize(using = CustomDeserialize.class)
        private LocalDate isFaxed;


        @JsonProperty("is_canceled")
        @JsonDeserialize(using = CustomDeserialize.class)
        private LocalDate isCanceled;

        @JsonProperty("contract_type")
        private Integer contractType;

        @JsonProperty("test_mode")
        private Integer testMode;

        @Override
        public LocalDate getDt() {
            return dt;
        }

        public void setDt(LocalDate dt) {
            this.dt = dt;
        }

        @Override
        public LocalDate getContractEndDt() {
            return contractEndDt;
        }

        public void setContractEndDt(LocalDate contractEndDt) {
            this.contractEndDt = contractEndDt;
        }

        @Override
        public LocalDate getIsSigned() {
            return isSigned;
        }

        public void setIsSigned(LocalDate isSigned) {
            this.isSigned = isSigned;
        }

        @Override
        public LocalDate getIsFaxed() {
            return isFaxed;
        }

        public void setIsFaxed(LocalDate isFaxed) {
            this.isFaxed = isFaxed;
        }

        @Override
        public LocalDate getIsCanceled() {
            return isCanceled;
        }

        public void setIsCanceled(LocalDate isCanceled) {
            this.isCanceled = isCanceled;
        }

        @Override
        public Integer getContractType() {
            return contractType;
        }

        public void setContractType(Integer contractType) {
            this.contractType = contractType;
        }

        @Override
        public Integer getTestMode() {
            return testMode;
        }

        public void setTestMode(Integer testMode) {
            this.testMode = testMode;
        }
    }

    private static class CustomDeserialize extends StdDeserializer<LocalDate> {
        private final LocalDateTimeDeserializer localDateTimeDeserializer =
                new LocalDateTimeDeserializer(DateTimeFormatter.ISO_LOCAL_DATE_TIME);

        private final String dateFormat = "yyyy-MM-dd";
        private final LocalDateDeserializer localDateDeserializer =
                new LocalDateDeserializer(DateTimeFormatter.ofPattern(dateFormat));

        protected CustomDeserialize() {
            super(LocalDate.class);
        }

        @Override
        public LocalDate deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
            JsonToken token = p.currentToken();
            var value = p.getValueAsString();
            if (value == null || value.isEmpty()) {
                return null;
            } else if (value.length() > dateFormat.length()) {
                return localDateTimeDeserializer.deserialize(p, ctxt).toLocalDate();
            } else {
                return localDateDeserializer.deserialize(p, ctxt);
            }

        }
    }
}
