package ru.yandex.partner.core.entity.user.multistate;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.user.actions.UserActionsEnum;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.messages.TextTemplateMsg;
import ru.yandex.partner.core.messages.UserActionMsg;
import ru.yandex.partner.core.multistate.Multistate;
import ru.yandex.partner.core.multistate.user.UserMultistate;
import ru.yandex.partner.core.multistate.user.UserStateFlag;
import ru.yandex.partner.libs.i18n.GettextMsg;
import ru.yandex.partner.libs.multistate.action.ActionCheckId;
import ru.yandex.partner.libs.multistate.action.ActionEntry;
import ru.yandex.partner.libs.multistate.action.ActionNameHolder;
import ru.yandex.partner.libs.multistate.graph.AbstractMultistateGraph;

import static java.util.function.Predicate.not;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.empty;
import static ru.yandex.partner.libs.multistate.MultistatePredicates.has;

@Component
@ParametersAreNonnullByDefault
public class UserMultistateGraph extends AbstractMultistateGraph<User, UserStateFlag> {

    private final UserActionChecksService userActionChecksService;

    @Autowired
    public UserMultistateGraph(UserActionChecksService userActionChecksService) {
        this.userActionChecksService = userActionChecksService;
    }

    @Override
    public Set<ModelProperty<? super User, ?>> getRequiredPropertiesByActionName(String actionName) {
        var actionEntry = this.getActionEntry(actionName);

        Set<ModelProperty<? super User, ?>> requiredProperties = new HashSet<>(actionEntry.getRequiredProperties());

        var requiredPropertiesFromChecks = actionEntry.getChecks().stream()
                .map(check -> userActionChecksService.getActionCheck(check).getRequiredModelProperties())
                .flatMap(Set::stream).collect(Collectors.toSet());

        requiredProperties.addAll(requiredPropertiesFromChecks);

        return requiredProperties;
    }

    @Override
    protected List<Boolean> performCheck(ActionCheckId check, List<? extends User> models) {
        if (!(check instanceof UserActionChecksService.UserActionCheck)) {
            return super.performCheck(check, models);
        }
        Function<List<? extends User>, List<Boolean>> checkFunction =
                userActionChecksService.getActionCheck(check).getCheck();
        return checkFunction == null ? super.performCheck(check, models)
                : checkFunction.apply(models);
    }

    @Override
    public UserMultistate getMultistateFromModel(User model) {
        return model.getMultistate();
    }

    @Override
    protected Multistate<UserStateFlag> getMultistateForValue(Long multistateValue) {
        return new UserMultistate(multistateValue);
    }

    @Override
    protected Map<ActionNameHolder, ActionEntry<User, UserStateFlag>> createGraph() {
        Map<ActionNameHolder, ActionEntry<User, UserStateFlag>> actionEntryMap = new HashMap<>();
        actionEntryMap.put(UserActionsEnum.ADD,
                getActionEntryBuilder(TextTemplateMsg.ADD)
                        .setPredicate(empty())
                        .build());

        actionEntryMap.put(UserActionsEnum.PROVIDE_CONTACTS,
                getActionEntryBuilder(UserActionMsg.PROVIDE_CONTACTS)
                        .setPredicate(not(has(UserStateFlag.CONTACTS_PROVIDED)))
                        .setSetFlags(Map.of(UserStateFlag.CONTACTS_PROVIDED, true))
                        .build());

        actionEntryMap.put(UserActionsEnum.REQUEST_CREATE_IN_BANNER_STORE,
                getActionEntryBuilder(UserActionMsg.REQUEST_CREATE_IN_BANNER_STORE)
                        .setPredicate(not(has(UserStateFlag.NEED_CREATE_IN_BANNER_STORE)))
                        .setSetFlags(Map.of(UserStateFlag.NEED_CREATE_IN_BANNER_STORE, true))
                        .build());

        actionEntryMap.put(UserActionsEnum.CREATED_PARTNER_IN_BANNER_STORE,
                getActionEntryBuilder(UserActionMsg.CREATED_PARTNER_IN_BANNER_STORE)
                        .setPredicate(has(UserStateFlag.NEED_CREATE_IN_BANNER_STORE))
                        .setSetFlags(Map.of(UserStateFlag.NEED_CREATE_IN_BANNER_STORE, false))
                        .build());

        actionEntryMap.put(UserActionsEnum.SET_USER_ROLE,
                getActionEntryBuilder(UserActionMsg.SET_USER_ROLE)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .build());

        actionEntryMap.put(UserActionsEnum.REVOKE_ROLES,
                getActionEntryBuilder(UserActionMsg.REVOKE_ROLES)
                        .build());

        actionEntryMap.put(UserActionsEnum.REQUEST_YAN_CONTRACT,
                getActionEntryBuilder(UserActionMsg.REQUEST_YAN_CONTRACT)
                        .setPredicate(not(has(UserStateFlag.NEED_YAN_CONTRACT)))
                        .setSetFlags(Map.of(UserStateFlag.NEED_YAN_CONTRACT, true))
                        .build());

        actionEntryMap.put(UserActionsEnum.YAN_CONTRACT_READY,
                getActionEntryBuilder(UserActionMsg.YAN_CONTRACT_READY)
                        .setPredicate(has(UserStateFlag.NEED_YAN_CONTRACT))
                        .setSetFlags(Map.of(UserStateFlag.NEED_YAN_CONTRACT, false))
                        .build());

        actionEntryMap.put(UserActionsEnum.LINK_ADFOX_USER,
                getActionEntryBuilder(UserActionMsg.LINK_ADFOX_USER)
                        .build());

        actionEntryMap.put(UserActionsEnum.UNLINK_ADFOX_USER,
                getActionEntryBuilder(UserActionMsg.UNLINK_ADFOX_USER)
                        .build());

        actionEntryMap.put(UserActionsEnum.UNSUBSCRIBE_FROM_STAT_MONITORING_EMAILS,
                getActionEntryBuilder(UserActionMsg.UNSUBSCRIBE_FROM_STAT_MONITORING_EMAILS)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .build());

        actionEntryMap.put(UserActionsEnum.EDIT,
                getActionEntryBuilder(TextTemplateMsg.EDIT)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .build());

        actionEntryMap.put(UserActionsEnum.SET_EXCLUDED_DOMAINS,
                getActionEntryBuilder(UserActionMsg.SET_EXCLUDED_DOMAINS)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .build());

        actionEntryMap.put(UserActionsEnum.SET_EXCLUDED_PHONES,
                getActionEntryBuilder(UserActionMsg.SET_EXCLUDED_PHONES)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .build());

        actionEntryMap.put(UserActionsEnum.SET_BLOCKED,
                getActionEntryBuilder(UserActionMsg.SET_BLOCKED)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .setSetFlags(Map.of(UserStateFlag.BLOCKED, true))
                        .build());

        actionEntryMap.put(UserActionsEnum.RESET_BLOCKED,
                getActionEntryBuilder(UserActionMsg.RESET_BLOCKED)
                        .setPredicate(has(UserStateFlag.BLOCKED))
                        .setSetFlags(Map.of(UserStateFlag.BLOCKED, false))
                        .build());

        actionEntryMap.put(UserActionsEnum.CHANGE_CONTRACT,
                getActionEntryBuilder(UserActionMsg.EDIT_REQUISITES)
                        .setPredicate(not(has(UserStateFlag.BLOCKED)))
                        .setChecks(List.of(UserActionChecksService.UserActionCheck.CAN_CHANGE_REQUISITES))
                        .setRequiredProperties(Set.of(User.CLIENT_ID))
                        .build());

        return actionEntryMap;
    }

    private ActionEntry.Builder<User, UserStateFlag> getActionEntryBuilder(GettextMsg titleMsg) {
        return getActionEntryBuilder(titleMsg, Set.of(User.ID, User.MULTISTATE));
    }

    @Override
    public Class<User> getModelClass() {
        return User.class;
    }

    @Override
    public Multistate<UserStateFlag> convertMultistate(List<UserStateFlag> enabledFlags) {
        return new UserMultistate(enabledFlags);
    }
}
