package ru.yandex.partner.core.entity.user.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.AppliedChangesValidatedStep;
import ru.yandex.direct.operation.update.ChangesAppliedStep;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.ModelChangesValidatedStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.partner.core.entity.user.container.UserAdditionalActionsContainer;
import ru.yandex.partner.core.entity.user.container.UserContainer;
import ru.yandex.partner.core.entity.user.container.UserContainerImpl;
import ru.yandex.partner.core.entity.user.container.UserRepositoryContainer;
import ru.yandex.partner.core.entity.user.model.BaseUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.repository.UserModifyRepository;
import ru.yandex.partner.core.entity.user.repository.UserTypedRepository;
import ru.yandex.partner.core.entity.user.service.type.update.UserUpdateOperationTypeSupportFacade;
import ru.yandex.partner.core.entity.user.service.validation.type.UserValidationTypeSupportFacade;
import ru.yandex.partner.core.multitype.service.validation.type.update.EditableFieldValidator;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class UserUpdateOperation extends SimpleAbstractUpdateOperation<BaseUser, Long> {
    private static final Logger LOGGER = LoggerFactory.getLogger(UserUpdateOperation.class);
    private final DSLContext dslContext;
    private final UserRepositoryContainer parametersContainer;
    private final UserContainer operationContainer;
    private final UserAdditionalActionsContainer additionalActionsContainer;
    private final UserTypedRepository typedRepository;
    private final UserModifyRepository modifyRepository;
    private final UserUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade;
    private final UserValidationTypeSupportFacade validationTypeSupportFacade;
    private final EditableFieldValidator<BaseUser> editableFieldValidator;

    @SuppressWarnings("checkstyle:parameternumber")
    public UserUpdateOperation(Applicability applicability,
                               List<ModelChanges<BaseUser>> modelChanges,
                               DSLContext dslContext,
                               UserTypedRepository typedRepository,
                               UserModifyRepository modifyRepository,
                               UserUpdateOperationTypeSupportFacade updateOperationTypeSupportFacade,
                               UserValidationTypeSupportFacade validationTypeSupportFacade,
                               EditableFieldValidator<BaseUser> editableFieldValidator) {
        super(applicability, modelChanges, new User()::withId);
        this.dslContext = dslContext;
        this.typedRepository = typedRepository;
        this.modifyRepository = modifyRepository;
        this.updateOperationTypeSupportFacade = updateOperationTypeSupportFacade;
        this.validationTypeSupportFacade = validationTypeSupportFacade;
        this.editableFieldValidator = editableFieldValidator;
        this.parametersContainer = new UserRepositoryContainer();
        this.additionalActionsContainer = new UserAdditionalActionsContainer();
        this.operationContainer = UserContainerImpl.create();
    }

    @Override
    protected Collection<BaseUser> getModels(Collection<Long> ids) {
        return typedRepository.getTyped(ids, true);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseUser>>, Defect> validateModelChanges(
            List<ModelChanges<BaseUser>> modelChanges) {
        ValidationResult<List<ModelChanges<BaseUser>>, Defect> vr = new ValidationResult<>(modelChanges);
        validationTypeSupportFacade.updateValidateModelChanges(operationContainer, vr);
        return vr;
    }

    @Override
    protected void onModelChangesValidated(ModelChangesValidatedStep<BaseUser> modelChangesValidatedStep) {
        Collection<ModelChanges<BaseUser>> modelChanges =
                modelChangesValidatedStep.getValidModelChanges();
        updateOperationTypeSupportFacade.onModelChangesValidated(operationContainer, modelChanges);
    }

    @Override
    protected ValidationResult<List<ModelChanges<BaseUser>>, Defect> validateModelChangesBeforeApply(
            ValidationResult<List<ModelChanges<BaseUser>>, Defect> preValidateResult, Map<Long, BaseUser> models) {
        editableFieldValidator.updateValidateBeforeApply(preValidateResult, models);
        validationTypeSupportFacade.updateValidateBeforeApply(operationContainer, preValidateResult, models);
        return preValidateResult;
    }

    @Override
    protected void onChangesApplied(ChangesAppliedStep<BaseUser> changesAppliedStep) {
        List<AppliedChanges<BaseUser>> validAppliedChanges =
                new ArrayList<>(changesAppliedStep.getAppliedChangesForValidModelChanges());
        updateOperationTypeSupportFacade.onChangesApplied(operationContainer, validAppliedChanges);
    }

    @Override
    protected ValidationResult<List<BaseUser>, Defect> validateAppliedChanges(ValidationResult<List<BaseUser>,
            Defect> validationResult, Map<Integer, AppliedChanges<BaseUser>> appliedChangesForValidModelChanges) {
        validationTypeSupportFacade.updateValidateAppliedChanges(operationContainer, validationResult,
                appliedChangesForValidModelChanges);
        if (!validationResult.hasAnyErrors()) {
            validationTypeSupportFacade.validate(operationContainer, validationResult);
        }
        return validationResult;
    }

    @Override
    protected void onAppliedChangesValidated(AppliedChangesValidatedStep<BaseUser> appliedChangesValidatedStep) {
        var validAppliedChanges = new ArrayList<>(appliedChangesValidatedStep.getValidAppliedChanges());

        updateOperationTypeSupportFacade.onAppliedChangesValidated(operationContainer, validAppliedChanges);
    }

    @Override
    protected void beforeExecution(ExecutionStep<BaseUser> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());

        updateOperationTypeSupportFacade.beforeExecution(appliedChangesForExecution, operationContainer);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<BaseUser>> applicableAppliedChanges) {
        updateOperationTypeSupportFacade.addToAdditionalActionsContainer(additionalActionsContainer,
                operationContainer,
                applicableAppliedChanges);
        processUpdateTask(applicableAppliedChanges);
        return mapList(applicableAppliedChanges, b -> b.getModel().getId());
    }

    private void processUpdateTask(List<AppliedChanges<BaseUser>> validAppliedChanges) {
        modifyRepository.update(dslContext, parametersContainer, validAppliedChanges);
    }

    @Override
    protected void afterExecution(ExecutionStep<BaseUser> executionStep) {
        var appliedChangesForExecution = new ArrayList<>(executionStep.getAppliedChangesForExecution());
        updateOperationTypeSupportFacade.afterExecution(appliedChangesForExecution, operationContainer);
    }
}
