package ru.yandex.partner.core.entity.user.type;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.springframework.stereotype.Component;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.partner.core.entity.user.container.UserContainer;
import ru.yandex.partner.core.entity.user.container.UserRepositoryContainer;
import ru.yandex.partner.core.entity.user.model.BaseUser;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.entity.user.service.type.update.AbstractUserUpdateOperationTypeSupport;
import ru.yandex.partner.core.multitype.repository.PartnerRepositoryTypeSupportFacade;
import ru.yandex.partner.core.role.Role;
import ru.yandex.partner.libs.memcached.MemcachedService;

/**
 * TypeSupport над объединяющим интерфейсом
 * Для инвалидации кеша нужно смотреть на OPTS и ROLES, а они в разных интерфейсах - поэтому здесь над User
 */
@Component
public class UserUpdateOperationTypeSupport extends AbstractUserUpdateOperationTypeSupport<User> {

    public static final String AVAILABLE_RESOURCES_CACHE_PREFIX = "available_resources";

    private final MemcachedService memcachedService;

    public UserUpdateOperationTypeSupport(MemcachedService memcachedService,
                                          PartnerRepositoryTypeSupportFacade<BaseUser, Long, UserRepositoryContainer,
            UserRepositoryContainer> repositoryFacade) {
        super(repositoryFacade);
        this.memcachedService = memcachedService;
    }


    @Override
    public Class<User> getTypeClass() {
        return User.class;
    }

    @Override
    public void afterExecution(UserContainer updateContainer, List<AppliedChanges<User>> appliedChanges) {
        super.afterExecution(updateContainer, appliedChanges);
        invalidateAvailableResourceCache(getCacheKeysToInvalidate(appliedChanges));
    }

    @Nonnull
    private Set<String> getCacheKeysToInvalidate(List<AppliedChanges<User>> appliedChanges) {
        return appliedChanges.stream()
                .filter(appliedChange -> appliedChange.getPropertiesForUpdate().contains(User.HAS_RSYA)
                        || appliedChange.getPropertiesForUpdate().contains(User.HAS_MOBILE_MEDIATION))
                .filter(this::isAvailableResourcesCacheNeedsInvalidation)
                .map(this::generateAvailableResourcesCacheKey)
                .collect(Collectors.toSet());
    }

    private boolean isAvailableResourcesCacheNeedsInvalidation(AppliedChanges<User> appliedChange) {
        return appliedChange.changed(User.HAS_RSYA) || appliedChange.changed(User.HAS_MOBILE_MEDIATION);
    }

    @Nonnull
    private String generateAvailableResourcesCacheKey(AppliedChanges<User> appliedChange) {
        Set<Role> roles = Optional.ofNullable(appliedChange.getOldValue(User.ROLES)).orElseGet(Set::of);
        String rolesString = roles.stream()
                .map(Role::getRoleId)
                .sorted()
                .map(String::valueOf)
                .collect(Collectors.joining(","));

        return "user_id: " + appliedChange.getModel().getId() + ", roles: [" + rolesString + "]";
    }

    private void invalidateAvailableResourceCache(Set<String> keys) {
        keys.forEach(key -> memcachedService.delete(AVAILABLE_RESOURCES_CACHE_PREFIX, key));
    }

}
