package ru.yandex.partner.core.entity.user.type.common;

import java.math.BigDecimal;
import java.util.Objects;
import java.util.Set;

import org.apache.commons.lang3.BooleanUtils;
import org.springframework.stereotype.Component;

import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.constraint.NumberConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;
import ru.yandex.partner.core.entity.user.defect.UserDefectIds;
import ru.yandex.partner.core.entity.user.model.CommonUser;
import ru.yandex.partner.core.validation.defects.PartnerCollectionDefects;
import ru.yandex.partner.core.validation.defects.ids.ModelPropertyDefectIds;
import ru.yandex.partner.core.validation.defects.params.ModelPropertyDefectParams;

import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.partner.core.validation.constraints.PartnerCommonConstraints.isValidEmailRfc822Pat;

@Component
public class CommonUserValidatorProvider {
    private static final Set<String> CPM_AVAILABLE_CURRENCIES = Set.of("USD", "RUB", "EUR");
    private static final Set<String> PAYONEER_AVAILABLE_CURRENCIES = Set.of("USD", "EUR");

    public <M extends CommonUser> Validator<M, Defect> validator() {
        // don't have constraints
        // newsletter
        // no_stat_monitoring_emails
        // has_tutby_agreement
        // is_dm_lite
        // is_efir_blogger
        // is_games
        // is_mobile_mediation
        // is_tutby
        // is_video_blogger
        // last_payout
        // need_to_email_processing
        return commonUser -> {
            var vb = ModelItemValidationBuilder.of(commonUser);
            vb.item(CommonUser.NAME)
                    .check(maxStringLength(255));
            vb.item(CommonUser.LASTNAME)
                    .check(maxStringLength(255));
            vb.item(CommonUser.MIDNAME)
                    .check(maxStringLength(255));
            vb.item(CommonUser.EMAIL)
                    .check(emailsStr -> {
                        if (emailsStr == null) {
                            return new Defect<>(UserDefectIds.CommonUser.INCORRECT_EMAIL);
                        }

                        String[] emails = emailsStr.split(",");
                        for (String email : emails) {
                            if (!isValidEmailRfc822Pat(email.trim())) {
                                return new Defect<>(UserDefectIds.CommonUser.INCORRECT_EMAIL);
                            }
                        }

                        return null;
                    });

            vb.item(CommonUser.PHONE)
                    .check(maxStringLength(255));


            // has_mobile_mediation
            vb.check(commonUserHasMobileMediation -> {
                boolean isMobileMediation = BooleanUtils.toBoolean(commonUser.getIsMobileMediation());
                boolean hasMobileMediation = BooleanUtils.toBoolean(commonUser.getHasMobileMediation());
                if (isMobileMediation && !hasMobileMediation) {
                    return new Defect<>(UserDefectIds.CommonUser.INCORRECT_HAS_MOBILE_MEDIATION);
                }

                return null;
            });

            // has_rsya
            vb.check(commonUserHasRsya -> {
                boolean isMobileMediation = BooleanUtils.toBoolean(commonUser.getIsMobileMediation());
                if (Objects.isNull(commonUser.getHasRsya())) {
                    return null;
                }
                boolean hasRsya = BooleanUtils.toBoolean(commonUser.getHasRsya());

                if (!isMobileMediation && !hasRsya) {
                    return new Defect<>(UserDefectIds.CommonUser.INCORRECT_HAS_RSYA);

                }
                return null;
            });

            vb.item(CommonUser.PAYONEER_CURRENCY)
                    .check(CommonConstraints.inSet(PAYONEER_AVAILABLE_CURRENCIES),
                            PartnerCollectionDefects.inSet(PAYONEER_AVAILABLE_CURRENCIES));

            vb.item(CommonUser.PAYONEER_PAYEE_ID)
                    .check(maxStringLength(30));

            vb.item(CommonUser.PAYONEER_STEP)
                    .check(NumberConstraints.notLessThan(0))
                    .check(NumberConstraints.notGreaterThan(8));

            vb.item(CommonUser.CURRENT_CURRENCY)
                    .check(CommonConstraints.inSet(CPM_AVAILABLE_CURRENCIES),
                            PartnerCollectionDefects.inSet(CPM_AVAILABLE_CURRENCIES));

            vb.item(CommonUser.NEXT_CURRENCY)
                    .check(CommonConstraints.inSet(CPM_AVAILABLE_CURRENCIES),
                            PartnerCollectionDefects.inSet(CPM_AVAILABLE_CURRENCIES));

            if (commonUser.getNextCurrency() != null &&
                    !commonUser.getNextCurrency().equals(commonUser.getCurrentCurrency())) {

                // если происходит смена валюты, важно чтобы было положительным числом
                vb.item(CommonUser.CURRENCY_RATE)
                        .check(CommonConstraints.notNull())
                        .check(
                                NumberConstraints.greaterThan(BigDecimal.ZERO),
                                new Defect<>(
                                        ModelPropertyDefectIds.FIELD_SHOULD_BE_POSITIVE_NUMBER,
                                        new ModelPropertyDefectParams(commonUser.getClass(), CommonUser.CURRENCY_RATE)
                                ),
                                When.isValid()
                        );
            }
            vb.item(CommonUser.INN)
                    .check(maxStringLength(12));

            return vb.getResult();
        };
    }
}
