package ru.yandex.partner.core.entity.user.type.cpm;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import NPartner.Page.TPartnerPage.TBlock;
import NPartner.Page.TPartnerPage.TBlock.TAdType;
import NPartner.Page.TPartnerPage.TBlock.TArticle;
import NPartner.Page.TPartnerPage.TBlock.TBrand;
import NPartner.Page.TPartnerPage.TBlock.TGeo;
import com.google.common.annotations.VisibleForTesting;
import org.springframework.util.CollectionUtils;

import ru.yandex.direct.model.ModelChanges;
import ru.yandex.partner.core.action.ActionPerformer;
import ru.yandex.partner.core.action.factories.ModelPayloadActionFactory;
import ru.yandex.partner.core.entity.QueryOpts;
import ru.yandex.partner.core.entity.block.actions.all.BlockActionEdit;
import ru.yandex.partner.core.entity.block.filter.BlockFilters;
import ru.yandex.partner.core.entity.block.model.BaseBlock;
import ru.yandex.partner.core.entity.block.model.BlockWithBrands;
import ru.yandex.partner.core.entity.block.model.BlockWithCustomBkData;
import ru.yandex.partner.core.entity.block.model.BlockWithGeo;
import ru.yandex.partner.core.entity.block.model.BlockWithMultistate;
import ru.yandex.partner.core.entity.block.model.BlockWithPiCategories;
import ru.yandex.partner.core.entity.block.model.BlockWithStrategy;
import ru.yandex.partner.core.entity.block.model.Brand;
import ru.yandex.partner.core.entity.block.model.Geo;
import ru.yandex.partner.core.entity.block.model.MobileRtbBlock;
import ru.yandex.partner.core.entity.block.model.PiCategory;
import ru.yandex.partner.core.entity.block.model.RtbBlock;
import ru.yandex.partner.core.entity.block.service.BlockService;
import ru.yandex.partner.core.entity.block.service.BlockUpdateOperation;
import ru.yandex.partner.core.entity.block.service.BlockUpdateOperationFactory;
import ru.yandex.partner.core.entity.block.type.custombkdata.BkDataException;
import ru.yandex.partner.core.entity.notification.UserNotificationsRepository;
import ru.yandex.partner.core.entity.user.actions.UserActionEdit;
import ru.yandex.partner.core.entity.user.actions.factories.UserEditFactory;
import ru.yandex.partner.core.entity.user.model.User;
import ru.yandex.partner.core.multistate.block.BlockStateFlag;
import ru.yandex.partner.core.service.mappers.MappersService;
import ru.yandex.partner.libs.bs.json.BkDataConverter;
import ru.yandex.partner.libs.multistate.graph.MultistateGraph;

import static java.math.RoundingMode.FLOOR;
import static ru.yandex.partner.core.CoreConstants.BK_MAX_CPM;
import static ru.yandex.partner.core.CoreConstants.CPM_MAX_SCALE;
import static ru.yandex.partner.core.CoreConstants.MAX_CPM;
import static ru.yandex.partner.core.CoreConstants.Strategies.MIN_CPM_STRATEGY_ID;
import static ru.yandex.partner.core.CoreConstants.Strategies.SEPARATE_CPM_STRATEGY_ID;

public class CpmConversion {
    private static final Set<Class<? extends BlockWithMultistate>> BLOCKS_WITH_CPM_CURRENCY = Set.of(
            RtbBlock.class, MobileRtbBlock.class
    );
    private final BkDataConverter bkDataConverter = new BkDataConverter();

    private final BlockService blockService;
    private final UserEditFactory userEditFactory;
    private final MappersService mappersService;
    private final BlockUpdateOperationFactory blockUpdateOperationFactory;
    private final Map<Class<?>, MultistateGraph<? extends BaseBlock, BlockStateFlag>> graphs;
    private final Map<
            Class<?>,
            ModelPayloadActionFactory<? extends BaseBlock, BlockActionEdit<? extends BlockWithMultistate>>
            > actionEditFactories;
    private final ActionPerformer actionPerformer;
    private final UserNotificationsRepository userNotificationsRepository;

    private final Long userId;
    private final String nextCurrency;
    private final BigDecimal conversionRate;

    @SuppressWarnings("parameternumber")
    protected CpmConversion(
            BlockService blockService,
            UserEditFactory userEditFactory,
            MappersService mappersService,
            BlockUpdateOperationFactory blockUpdateOperationFactory,
            Map<Class<?>, MultistateGraph<? extends BaseBlock, BlockStateFlag>> graphs,
            Map<
                    Class<?>,
                    ModelPayloadActionFactory<? extends BaseBlock, BlockActionEdit<? extends BlockWithMultistate>>
                    > actionEditFactories,
            ActionPerformer actionPerformer,
            UserNotificationsRepository userNotificationsRepository,
            Long userId,
            String nextCurrency,
            BigDecimal conversionRate
    ) {
        this.blockService = blockService;
        this.userEditFactory = userEditFactory;
        this.mappersService = mappersService;
        this.blockUpdateOperationFactory = blockUpdateOperationFactory;
        this.graphs = graphs;
        this.actionEditFactories = actionEditFactories;
        this.actionPerformer = actionPerformer;
        this.userNotificationsRepository = userNotificationsRepository;

        this.userId = userId;
        this.nextCurrency = nextCurrency;
        this.conversionRate = conversionRate;
    }

    /**
     * Конвертирует cpm для всех блоков данного юзера,
     * редактирует валюту у пользователя и создает уведомление про это
     */
    public void convert() {
        BLOCKS_WITH_CPM_CURRENCY.forEach(c -> {
            var models = blockService.findAll(QueryOpts.forClass(c)
                    .withFilter(BlockFilters.OWNER_ID.eq(userId))
                    .forUpdate(true)
            );
            models.forEach(this::convertBlock);
        });
        UserActionEdit actionEdit = userEditFactory.edit(new ModelChanges<>(userId, User.class)
                // todo: в перле не редактируется currency_rate на юзере
                .process(nextCurrency, User.CURRENT_CURRENCY)
        );
        User user = actionPerformer.doActions(actionEdit).getResult(User.class).stream()
                .findFirst().orElseThrow()
                .get(0)
                .getResult();
        userNotificationsRepository.createCpmConvertNotification(user);
    }

    @VisibleForTesting
    @SuppressWarnings({"unchecked", "rawtypes"})
    public <T extends BaseBlock> void convertBlock(T block) {
        // осознанно используем raw type, так как в changes будут поля от разных типов блока
        ModelChanges changes = new ModelChanges<>(block.getId(), block.getClass());

        boolean isCustomBkData = block instanceof BlockWithCustomBkData blockWithCustomBkData &&
                Boolean.TRUE.equals(blockWithCustomBkData.getIsCustomBkData());

        if (isCustomBkData) {
            convertBkData(changes, (BlockWithCustomBkData) block);
        } else {
            convertPiCategories(changes, block);
            convertBrands(changes, block);
            convertGeo(changes, block);
            convertStrategy(changes, block);
        }

        boolean isEditAllowed = ((MultistateGraph<T, BlockStateFlag>) graphs.get(block.getClass()))
                .checkActionAllowed("edit", block);

        if (isEditAllowed) {
            BlockActionEdit action = actionEditFactories.get(block.getClass()).createAction(List.of(changes));
            actionPerformer.doActions(action);
        } else {
            BlockUpdateOperation operation = blockUpdateOperationFactory.createWithModelChangesOnly(List.of(changes));
            operation.prepareAndApply();
        }
    }

    private void convertPiCategories(ModelChanges changes, BaseBlock block) {
        if (block instanceof BlockWithPiCategories blockWithPiCategories &&
                !CollectionUtils.isEmpty(blockWithPiCategories.getPiCategories())) {
            List<PiCategory> categories = blockWithPiCategories.getPiCategories().stream()
                    .map(mappersService.piCategoryMapper()::clone)
                    .toList();
            categories.forEach(c -> c.setCpm(convertValue(c.getCpm())));
            changes.process(categories, BlockWithPiCategories.PI_CATEGORIES);
        }
    }

    private void convertBrands(ModelChanges changes, BaseBlock block) {
        if (block instanceof BlockWithBrands blockWithBrands &&
                !CollectionUtils.isEmpty(blockWithBrands.getBrands())) {
            List<Brand> brands = blockWithBrands.getBrands().stream()
                    .map(mappersService.brandMapper()::clone)
                    .toList();
            brands.forEach(b -> b.setCpm(convertValue(b.getCpm())));
            changes.process(brands, BlockWithBrands.BRANDS);
        }
    }

    private void convertGeo(ModelChanges changes, BaseBlock block) {
        if (block instanceof BlockWithGeo blockWithGeo && blockWithGeo.getGeo() != null &&
                !CollectionUtils.isEmpty(blockWithGeo.getGeo())) {
            List<Geo> geos = blockWithGeo.getGeo().stream()
                    .map(mappersService.geoMapper()::clone)
                    .toList();
            geos.forEach(g -> g.setCpm(convertValue(g.getCpm())));
            changes.process(geos, BlockWithGeo.GEO);
        }
    }

    private void convertStrategy(ModelChanges changes, BaseBlock block) {
        if (block instanceof BlockWithStrategy blockWithStrategy) {
            if (blockWithStrategy.getStrategyType().equals(MIN_CPM_STRATEGY_ID)) {
                changes.process(convertValue(blockWithStrategy.getMincpm()), BlockWithStrategy.MINCPM);
            } else if (blockWithStrategy.getStrategyType().equals(SEPARATE_CPM_STRATEGY_ID)) {
                changes.process(convertValue(blockWithStrategy.getTextCpm()), BlockWithStrategy.TEXT_CPM)
                        .process(convertValue(blockWithStrategy.getMediaCpm()), BlockWithStrategy.MEDIA_CPM)
                        .process(convertValue(blockWithStrategy.getVideoCpm()), BlockWithStrategy.VIDEO_CPM);
            }
        }
    }

    private void convertBkData(ModelChanges changes, BlockWithCustomBkData blockWithCustomBkData) {
        try {
            TBlock tBlock = bkDataConverter.convertBlockJsonToProto(blockWithCustomBkData.getBkData())
                    .getMessage();

            List<TBlock.TDSPInfo> tDspInfos = tBlock.getDSPInfoList().stream()
                    .map(i -> {
                        boolean isNeedConvert = i.hasCPM() && i.getCPM() > 0 &&
                                BigDecimal.valueOf(i.getCPM()).compareTo(BK_MAX_CPM) < 0;
                        if (isNeedConvert) {
                            return TBlock.TDSPInfo.newBuilder(i)
                                    .setCPM(convertValue(i.getCPM()))
                                    .build();
                        } else {
                            return i;
                        }
                    }).toList();

            List<TBlock.TPICategoryIAB> categoriesIAB = tBlock.getPICategoryIABList().stream()
                    .map(c -> TBlock.TPICategoryIAB.newBuilder(c)
                            // SHOULD MATCH https://wiki.yandex-team.ru/users/surch/ProductType/
                            // CPM
                            .setMediaCreativeReach(convertValue(c.getMediaCreativeReach()))
                            .setMediaImageReach(convertValue(c.getMediaImageReach()))
                            .setVideoCreativeReach(convertValue(c.getVideoCreativeReach()))
                            // CPC
                            .setMediaSmart(convertValue(c.getMediaSmart()))
                            .setMediaImage(convertValue(c.getMediaImage()))
                            .setMediaCreative(convertValue(c.getMediaCreative()))
                            .build()
                    ).toList();

            TBlock convertedBlock = TBlock.newBuilder(tBlock)
                    .setCPMCurrency(nextCurrency)
                    .clearDSPInfo()
                    .addAllDSPInfo(tDspInfos)
                    .clearPICategoryIAB()
                    .addAllPICategoryIAB(categoriesIAB)
                    .clearGeo()
                    .addAllGeo(convertedSimpleCpmObjects(tBlock.getGeoList(), TGeo.class))
                    .clearBrand()
                    .addAllBrand(convertedSimpleCpmObjects(tBlock.getBrandList(), TBrand.class))
                    .clearAdType()
                    .addAllAdType(convertedSimpleCpmObjects(tBlock.getAdTypeList(), TAdType.class))
                    .clearArticle()
                    .addAllArticle(convertedSimpleCpmObjects(tBlock.getArticleList(), TArticle.class))
                    .build();
            changes.process(bkDataConverter.convertProtoToJson(convertedBlock), BlockWithCustomBkData.BK_DATA);
        } catch (IOException e) {
            throw new BkDataException("Can't convert bkdata for block %s-%s: %s".formatted(
                    blockWithCustomBkData.getId(), blockWithCustomBkData.getPageId(), blockWithCustomBkData.getBkData()
            ), e);
        }
    }

    /**
     * Генерализуем вот такую конструкцию
     * <pre>
     * List<TGeo> tGeos = tBlock.getGeoList().stream()
     *     .map(g -> TGeo.newBuilder(g)
     *         .setValue(BigDecimal.valueOf(g.getValue()).divide(rate, FLOOR).longValue())
     *         .build()
     *     ).toList();
     * </pre>
     */
    @VisibleForTesting
    public <T> List<T> convertedSimpleCpmObjects(List<T> list, Class<T> clazz) {
        List<T> result = new ArrayList<>(list.size());
        try {
            for (T el : list) {
                // geoBuilder = TGeo.getBuilder(geo)
                Object builder = clazz.getMethod("newBuilder", clazz).invoke(null, el);

                // value = geo.getValue()
                long value = (long) clazz.getMethod("getValue").invoke(el);

                // geoBuilder.setValue( ... )
                builder.getClass()
                        .getMethod("setValue", long.class)
                        .invoke(builder, convertValueWithoutCheck(value));

                // TODO: нужно ли делать .setCurrency(nextCurrency) ? В перле такого нет

                // result.add( geoBuilder.build() )
                result.add((T) builder.getClass().getMethod("build").invoke(builder));
            }
        } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
            throw new RuntimeException("Can't convert cpm value for %s".formatted(clazz.getName()), e);
        }
        return result;
    }

    private Integer convertValue(Integer cpm) {
        if (cpm != null && cpm > 0 && BigDecimal.valueOf(cpm).compareTo(MAX_CPM) < 0) {
            return BigDecimal.valueOf(cpm).divide(conversionRate, CPM_MAX_SCALE, FLOOR).intValue();
        }
        return cpm;
    }

    private Long convertValueWithoutCheck(Long cpm) {
        return Optional.ofNullable(cpm)
                .map(BigDecimal::valueOf)
                .map(c -> c.divide(conversionRate, CPM_MAX_SCALE, FLOOR))
                .map(BigDecimal::longValue)
                .orElse(cpm);
    }

    private BigDecimal convertValue(BigDecimal cpm) {
        if (cpm != null && cpm.compareTo(BigDecimal.ZERO) > 0 && cpm.compareTo(MAX_CPM) < 0) {
            return cpm.divide(conversionRate, CPM_MAX_SCALE, FLOOR);
        }
        return cpm;
    }

}
