package ru.yandex.partner.core.entity.user.type.excludeddomains;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.user.container.UserRepositoryContainer;
import ru.yandex.partner.core.entity.user.model.UserWithExcludedDomains;
import ru.yandex.partner.core.entity.user.repository.type.AbstractUserRepositoryTypeSupport;
import ru.yandex.partner.core.entity.user.repository.type.UserRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;

import static ru.yandex.partner.core.entity.user.model.prop.UserWithExcludedDomainsExcludedDomainsPropHolder.EXCLUDED_DOMAINS;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.core.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class UserWithExcludedDomainsRepositoryTypeSupport
        extends AbstractUserRepositoryTypeSupport<UserWithExcludedDomains>
        implements UserRepositoryTypeSupportWithoutMapper<UserWithExcludedDomains> {

    private final UserWithExcludedDomainsRepositoryHelper excludedDomainsHelper;
    private final Set<ModelProperty<? super UserWithExcludedDomains, ?>> affectedModelProperties;

    public UserWithExcludedDomainsRepositoryTypeSupport(
            DSLContext dslContext,
            UserWithExcludedDomainsRepositoryHelper excludedDomainsHelper
    ) {
        super(dslContext);
        this.excludedDomainsHelper = excludedDomainsHelper;
        this.affectedModelProperties = Set.of(EXCLUDED_DOMAINS);
    }

    @Override
    public Set<ModelProperty<? super UserWithExcludedDomains, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(UserWithExcludedDomains model) {
        return fromModelProperties(new HashSet<>(affectedModelProperties));
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<UserWithExcludedDomains> models) {
        Set<Long> userIds = listToSet(models, UserWithExcludedDomains::getId);
        Map<Long, List<String>> excludedDomainsByUser = excludedDomainsHelper.groupByIds(userIds);
        models.forEach(userWithExcludedDomains ->
                userWithExcludedDomains.setExcludedDomains(
                        excludedDomainsByUser.getOrDefault(userWithExcludedDomains.getId(), List.of())));
    }

    @Override
    public void updateAdditionTables(DSLContext context,
                                     UserRepositoryContainer updateParameters,
                                     Collection<AppliedChanges<UserWithExcludedDomains>> appliedChanges) {
        excludedDomainsHelper.replaceValues(
                appliedChanges.stream()
                        .filter(change -> change.changed(EXCLUDED_DOMAINS))
                        .collect(Collectors.toMap(
                                change -> change.getModel().getId(),
                                change -> change.getNewValue(EXCLUDED_DOMAINS)
                        ))
        );
    }

    @Override
    public Class<UserWithExcludedDomains> getTypeClass() {
        return UserWithExcludedDomains.class;
    }
}
