package ru.yandex.partner.core.entity.user.type.features;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.partner.core.entity.user.container.UserRepositoryContainer;
import ru.yandex.partner.core.entity.user.model.UserWithFeatures;
import ru.yandex.partner.core.entity.user.repository.type.AbstractUserRepositoryTypeSupport;
import ru.yandex.partner.core.entity.user.repository.type.UserRepositoryTypeSupportWithoutMapper;
import ru.yandex.partner.core.holder.ModelPropertiesHolder;

import static ru.yandex.partner.core.entity.user.model.prop.UserWithFeaturesFeaturesPropHolder.FEATURES;
import static ru.yandex.partner.core.holder.ModelPropertiesHolder.fromModelProperties;
import static ru.yandex.partner.core.utils.FunctionalUtils.listToSet;

@Component
@ParametersAreNonnullByDefault
public class UserWithFeaturesRepositoryTypeSupport extends AbstractUserRepositoryTypeSupport<UserWithFeatures>
        implements UserRepositoryTypeSupportWithoutMapper<UserWithFeatures> {
    private final UserWithFeaturesRepositoryHelper repository;
    private final Set<ModelProperty<? super UserWithFeatures, ?>> affectedModelProperties;

    @Autowired
    protected UserWithFeaturesRepositoryTypeSupport(DSLContext dslContext,
                                                    UserWithFeaturesRepositoryHelper repository) {
        super(dslContext);
        this.repository = repository;
        this.affectedModelProperties = Set.of(FEATURES);
    }

    @Override
    public Set<ModelProperty<? super UserWithFeatures, ?>> getAffectedModelProperties() {
        return affectedModelProperties;
    }

    @Override
    public ModelPropertiesHolder getEditableModelProperties(UserWithFeatures model) {
        return fromModelProperties(new HashSet<>(affectedModelProperties));
    }

    @Override
    public void insertToAdditionTables(DSLContext context, UserRepositoryContainer addModelParametersContainer,
                                       Collection<UserWithFeatures> models) {
        repository.replaceValues(models.stream().collect(Collectors.toMap(
                UserWithFeatures::getId,
                UserWithFeatures::getFeatures
        )));
    }

    @Override
    public void updateAdditionTables(DSLContext context, UserRepositoryContainer updateParameters,
                                     Collection<AppliedChanges<UserWithFeatures>> appliedChanges) {
        repository.replaceValues(appliedChanges.stream()
                .filter(changes -> changes.changed(FEATURES))
                .map(AppliedChanges::getModel)
                .collect(Collectors.toMap(
                        UserWithFeatures::getId,
                        UserWithFeatures::getFeatures
                )));
    }

    @Override
    public void enrichModelFromOtherTables(DSLContext dslContext, Collection<UserWithFeatures> models) {
        Set<Long> userIds = listToSet(models, UserWithFeatures::getId);
        Map<Long, List<String>> featuresByUsers = repository.groupByIds(userIds);
        models.forEach(user -> user.setFeatures(featuresByUsers.get(user.getId())));
    }

    @Override
    public Class<UserWithFeatures> getTypeClass() {
        return UserWithFeatures.class;
    }
}
