package ru.yandex.partner.core.filter;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.validation.constraints.NotNull;

import org.jooq.Condition;
import org.jooq.impl.DSL;

import ru.yandex.partner.core.filter.container.ModelFilterContainer;
import ru.yandex.partner.core.filter.meta.MetaFilter;
import ru.yandex.partner.core.filter.operator.BinaryOperator;
import ru.yandex.partner.core.filter.operator.FilterOperator;
import ru.yandex.partner.core.filter.operator.UnaryOperator;
import ru.yandex.partner.core.utils.CollectionUtils;

public class CoreFilterNode<M> {
    private final BinaryOperator binaryOperator;
    private final List<CoreFilterNode<? super M>> filterNodes;
    private final CoreFilter<M> coreFilter;
    private final List<UnaryOperator> unaryOperators;
    private static final CoreFilterNode<?> NEUTRAL = new CoreFilterNode<>(new NeutralCoreFilter<>());

    private CoreFilterNode(BinaryOperator binaryOperator, List<CoreFilterNode<? super M>> filterNodes,
                           CoreFilter<M> coreFilter, List<UnaryOperator> unaryOperators) {
        this.binaryOperator = binaryOperator;
        this.filterNodes = filterNodes;
        this.coreFilter = coreFilter;
        this.unaryOperators = unaryOperators;
    }

    public CoreFilterNode(BinaryOperator binaryOperator, List<CoreFilterNode<? super M>> filterNodes) {
        this(Objects.requireNonNull(binaryOperator),
                CollectionUtils.checkNotNullAndNotEmpty(filterNodes),
                null,
                Collections.emptyList());
    }

    public CoreFilterNode(CoreFilter<M> coreFilter) {
        this(null,
                null,
                Objects.requireNonNull(coreFilter),
                Collections.emptyList());
    }

    public CoreFilterNode<M> addFilterNode(BinaryOperator binaryOperator, CoreFilterNode<M> node) {
        return new CoreFilterNode<>(binaryOperator, List.of(this, node));
    }

    public CoreFilterNode<M> addUnaryOperator(UnaryOperator unaryOperator) {
        var list = new ArrayList<>(unaryOperators);
        list.add(unaryOperator);
        return new CoreFilterNode<>(binaryOperator, filterNodes, coreFilter, Collections.unmodifiableList(list));
    }

    public Condition toCondition(Class<M> searchClass, ModelFilterContainer<? super M> filterContainer) {
        if (isValueNode()) {
            return doUnaryOperations(coreFilter.toCondition(searchClass, filterContainer));
        } else {
            Condition sumCondition = DSL.noCondition();
            for (CoreFilterNode filterNode : filterNodes) {
                Condition curCondition = filterNode.toCondition(searchClass, filterContainer);
                sumCondition = binaryOperator.doOperation(sumCondition, curCondition);
            }
            return doUnaryOperations(sumCondition);
        }
    }

    @Override
    public String toString() {
        return isValueNode() ? coreFilter.toString() : nonValueNodeToString();
    }

    private String nonValueNodeToString() {
        return filterNodes.stream()
                .map(f -> {
                    if (f.isValueNode() && f.coreFilter instanceof CoreFilterWithValues<?, ?> vf) {
                        return vf.toString();
                    }
                    return f.toString();
                })
                .collect(Collectors.joining(" " + binaryOperator + " ", "(", ")"));
    }


    private Condition doUnaryOperations(Condition condition) {
        Condition unaryCondition = condition;
        for (UnaryOperator unaryOperator : unaryOperators) {
            unaryCondition = unaryOperator.doOperation(unaryCondition);

        }
        return unaryCondition;
    }

    private boolean isValueNode() {
        return coreFilter != null;
    }

    public static <M, V> CoreFilterNode<M> create(MetaFilter<? super M, V> metaFilter,
                                                  FilterOperator filterOperator,
                                                  Collection<V> values) {
        return new CoreFilterNode<>(new CoreFilterWithValues<>(metaFilter, filterOperator, values));
    }

    public static <M, V> CoreFilterNode<M> create(MetaFilter<? super M, V> metaFilter,
                                                  FilterOperator filterOperator,
                                                  V value) {
        return create(metaFilter, filterOperator, List.of(value));
    }

    /**
     * @deprecated {@link MetaFilter#eq(Object)}}
     */
    @Deprecated
    public static <M, V> CoreFilterNode<M> eq(MetaFilter<? super M, V> metaFilter, V value) {
        return CoreFilterNode.create(metaFilter, FilterOperator.EQUALS, value);
    }

    /**
     * @deprecated {@link MetaFilter#in(Collection)}}
     */
    @Deprecated
    public static <M, V> CoreFilterNode<M> in(MetaFilter<? super M, V> metaFilter, Collection<V> values) {
        return CoreFilterNode.create(metaFilter, FilterOperator.IN, values);
    }

    /**
     * @deprecated {@link MetaFilter#notIn(Collection)}
     */
    @Deprecated
    public static <M, V> CoreFilterNode<M> notIn(MetaFilter<? super M, V> metaFilter, Collection<V> values) {
        return CoreFilterNode.create(metaFilter, FilterOperator.NOT_IN, values);
    }

    @SafeVarargs
    public static <M> CoreFilterNode<M> and(CoreFilterNode<? super M>... filters) {
        return CoreFilterNode.<M>and(List.of(filters));
    }

    public static <M> CoreFilterNode<M> and(List<CoreFilterNode<? super M>> filterList) {
        return new CoreFilterNode<>(BinaryOperator.AND, filterList);
    }

    @SafeVarargs
    public static <M> CoreFilterNode<M> or(CoreFilterNode<? super M>... filters) {
        return CoreFilterNode.<M>or(List.of(filters));
    }

    public static <M> CoreFilterNode<M> or(List<CoreFilterNode<? super M>> filterList) {
        return new CoreFilterNode<>(BinaryOperator.OR, filterList);
    }

    /**
     * @deprecated {@link MetaFilter#match(Object)}
     */
    @Deprecated
    public static <M, V> CoreFilterNode<M> match(MetaFilter<? super M, CoreFilterNode<V>> metaFilter,
                                                 CoreFilterNode<V> subFilter) {
        return CoreFilterNode.create(metaFilter, FilterOperator.MATCH, subFilter);
    }

    public CoreFilterNode<M> and(@NotNull CoreFilterNode<? super M> other) {
        return new CoreFilterNode<>(BinaryOperator.AND, List.of(this, other));
    }

    /**
     * Нейтральный CoreFilterNode, который следует использовать вместо null.
     *
     * @param <M> тип модели
     * @see NeutralCoreFilter
     */
    public static <M> CoreFilterNode<M> neutral() {
        return (CoreFilterNode<M>) NEUTRAL;
    }

    /**
     * @see AlwaysTrueCoreFilter
     */
    public static <M> CoreFilterNode<M> alwaysTrue() {
        return new CoreFilterNode<>(new AlwaysTrueCoreFilter<>());
    }

    /**
     * @see AlwaysFalseCoreFilter
     */
    public static <M> CoreFilterNode<M> alwaysFalse() {
        return new CoreFilterNode<>(new AlwaysFalseCoreFilter<>());
    }
}
