package ru.yandex.partner.core.filter.container;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.partner.core.filter.db.DbFilter;
import ru.yandex.partner.core.filter.meta.MetaFilter;

public abstract class CommonModelFilterContainer<M> implements ModelFilterContainer<M> {
    private final Map<MetaFilter<? extends M, ?>, List<DbFilter<?, ?>>> filterMap;

    @Autowired
    public CommonModelFilterContainer(List<DbFilter<? extends M, ?>> dbFilters) {
        filterMap = dbFilters.stream()
                .collect(Collectors.groupingBy(dbFilter -> (MetaFilter<? extends M, ?>) dbFilter.getMetaFilter()));
    }

    @Override
    public <E extends M, V> DbFilter<E, V> getDbFilter(Class<? super E> searchClass, MetaFilter<E, V> metaFilter) {
        if (metaFilter instanceof DbFilter dbFilter) {
            //noinspection unchecked
            return dbFilter;
        }

        List<DbFilter<?, ?>> dbFilters = filterMap.getOrDefault(metaFilter, List.of()).stream()
                .filter(dbFilter -> dbFilter.getResolvableModelClass().isAssignableFrom(searchClass))
                .toList();

        if (dbFilters.isEmpty()) {
            throw new IllegalArgumentException("Not found suitable DbFilter for " + metaFilter
                    + " for " + searchClass);
        } else if (dbFilters.size() == 1) {
            return (DbFilter<E, V>) dbFilters.get(0);
        } else {
            throw new IllegalArgumentException("Found more than one suitable DbFilter for " + metaFilter
                    + " for " + searchClass + "\n Matched filters: " + dbFilters);
        }
    }
}
